import { useTranslation } from 'react-i18next';
import { Button } from "@/components/shadcn/button";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '@/components/shadcn/dropdown-menu';
import EmailIcon from '@mui/icons-material/Email';
import { EmailReportDialog } from '@/dashboard/reports/EmailReportDialog';
import { useQuery } from '@tanstack/react-query';
import axios from 'axios';
import { useState } from 'react';
import { Loader2 } from "lucide-react"

export const ReportToolbar = ({ dashboardId, dashboardName }: { dashboardId: string, dashboardName: string }) => {
    const { t } = useTranslation();
    const [isGenerating, setIsGenerating] = useState(false);
    const [filename, setFilename] = useState('');
    const [type, setType] = useState('');

    const attachmentQuery = useQuery({
      queryKey: ['downloadReport'],
      queryFn: async () => {
        const response = await axios.get(`${window.location.origin}/nagioslogserver/reports/check_attachment/${filename}`);
        if (response.data.status === 'success') {
          window.open(`${window.location.origin}/nagioslogserver/reports/view_attachment/${filename}/${type}`, '_self');
          setIsGenerating(false);
        } else {
          console.log(t('Failed to find report.'));
        }
        return response.data;
      },
      enabled: false,
      retry: true,
      retryDelay: 2000
    });

    function exportReport(type: string) {
      const name = `report-${Math.floor(Date.now() / 1000)}.${type}`;
      const url = `${window.location.origin}/nagioslogserver/reports/view/${dashboardId}?export=${type}&filename=${name}`;
      setType(type);
      setFilename(name);
      setIsGenerating(true);

      fetch(url)
        .then(response => {
          if (!response.ok) {
            throw new Error(t('Error during report export.'));
          }
          return response;
        })
        .then(() => {
          attachmentQuery.refetch();
        })
        .catch(error => {
          console.log(t('Error during report export:'), error);
        })
    }

    const Overlay = () => {
      return (
        <div className="fixed inset-0 z-50 bg-black/80">
          <div className="relative top-[45%] left-1/2 z-[51] flex items-center gap-2 text-background dark:text-foreground">
            <Loader2 className="animate-spin" />
            {type === 'pdf' ? t("Generating PDF")
            : t("Generating JPG")
            }
          </div>
        </div>
      )
    }

    return (
      <div className="w-full flex p-4 justify-between items-center border-b border-border">
        {isGenerating &&
          <Overlay />
        }
        {dashboardName}
        <div className="flex gap-2 items-center justify-center">
          <EmailReportDialog dashboardName={dashboardName} dashboardId={dashboardId}>
            <Button variant="secondary" className="gap-2 border border-border print:hidden">
              <EmailIcon />
              {t("Email")}
            </Button>
          </EmailReportDialog>
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="secondary" className="border border-border flex items-center print:hidden">
                {t("Download")}
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent>
              <DropdownMenuItem onClick={() => exportReport("pdf")}>
                PDF
              </DropdownMenuItem>
              <DropdownMenuItem onClick={() => exportReport("jpg")}>
                JPG
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
        </div>
      </div>
    )
  
  }