import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/shadcn/dialog';
import { Input } from '@/components/shadcn/input';
import { Button } from '@/components/shadcn/button';
import { ReactNode, useState } from 'react';
import { useForm, SubmitHandler } from 'react-hook-form';
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/shadcn/form';
import { useTranslation } from 'react-i18next';
import { row } from '@/lib/types';
import { dashboardStore } from '@/dashboard/Dashboard';
import { toast } from '@/components/shadcn/use-toast';

/**
 * The row dialog. Passing in a row makes it an edit row dialog. Otherwise it's an add row dialog.
 */
export const AddRowDialog = ({row, children }: { row?: row, children: ReactNode }) => {
    const { definition, setDefinition } = dashboardStore();
    const [open, setOpen] = useState(false);
    const { t } = useTranslation();
    // const auth = useAuth();
    const form = useForm<row>({
        defaultValues: {
            title: row ? row.title : '',
        },
    });

    const addRow = (title: string) => {
        if (title) {
            definition.rows.push({id: `row${Date.now()}`, title: title, panels: []});
            setDefinition({...definition, rows: definition.rows});
            setOpen(false);
            toast({
                title: t('Row Added.'),
                variant: 'success',
                description: t('Add some panels to your new row!'),
            });
        } else {
            toast({
                title: t('Failed to add row.'),
                variant: 'error',
                description: t('Invalid title.'),
            });
        }
    }

    const editRow = (title: string) => {
        title = title;
        setOpen(false);
    }

    // I'm trying really hard to reuse this component for updating as well as adding dashboards, but they have different API endpoints - DA
    // I don't quite know what the comment above this means, but I guess the same but replace 'dashboards' with 'tools' - GW
    const onSubmit: SubmitHandler<row> = (data) => {
        if (row) {
            editRow(data.title);
        } else {
            addRow(data.title);
        }
    };

    return (
        <Dialog open={open} onOpenChange={setOpen}>
            <DialogTrigger asChild>
                <div>{children}</div>
            </DialogTrigger>
            <DialogContent className="sm:max-w-[425px]">
                <DialogHeader>
                    <DialogTitle>{row ? t('Edit Row') : t('Add Row')}</DialogTitle>
                    {!row && <DialogDescription>{t('Use this to add a new row to your dashboard')}</DialogDescription>}
                </DialogHeader>
                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-2">
                        <FormField
                            control={form.control}
                            name="title"
                            render={({ field }) => (
                                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                                    <FormLabel className="text-right">{t('Name')}</FormLabel>
                                    <FormControl>
                                        <Input {...field} className="col-span-3 border-border" />
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <DialogFooter>
                            <Button type="submit" className="justify-end">
                                {t('Submit')}
                            </Button>
                        </DialogFooter>
                    </form>
                </Form>
            </DialogContent>
        </Dialog>
    );
};
