import { useTranslation } from 'react-i18next';
import { dashboardStore } from '../Dashboard';
import { Button } from "@/components/shadcn/button";
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
} from '@/components/shadcn/tooltip';
import { toast } from '@/components/shadcn/use-toast';
import AddIcon from '@mui/icons-material/Add';
import ShareOutlinedIcon from '@mui/icons-material/ShareOutlined';
import ChatOutlinedIcon from '@mui/icons-material/ChatOutlined';
import NotificationAddOutlinedIcon from '@mui/icons-material/NotificationAddOutlined';
import SaveOutlinedIcon from '@mui/icons-material/SaveOutlined';
import ArticleOutlinedIcon from '@mui/icons-material/ArticleOutlined';
import IosShareIcon from '@mui/icons-material/IosShare';
import { SaveAs } from '@/dashboard/toolbar/SaveAs';
import { RefreshRateDropdown } from '@/dashboard/toolbar/RefreshRateDropdown';
import { AddRowDialog } from '@/dashboard/toolbar/AddRowDialog';
import { ManageQueriesDialog } from '@/dashboard/queries/ManageQueriesDialog';
import { AddAlertDialog } from '@/dashboard/alerts/AddAlertDialog';
import { ShareDashboardDialog } from '@/dashboard/toolbar/ShareDashboardDialog';
import { TimePicker } from '@/dashboard/toolbar/TimePicker';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { AxiosError } from 'axios';
import { editDashboard } from '@/dashboard/queries';
import { addReport } from '@/dashboard/reports/queries';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import { useSidebar } from "@/components/shadcn/sidebar";
import { PanelLeft } from 'lucide-react';
import { ExportDashboardDialog } from '../ExportDashboardDialog';
import { ImportDashboardDialog } from '../ImportDashboardDialog';

export const DashboardToolbar = ({ dashboardId, dashboardName }: { dashboardId: string, dashboardName: string }) => {
    const { definition } = dashboardStore();
    const { filters, subQueries, range } = useDashboardContext().dashboardInfo;
    const { t } = useTranslation();
    const queryClient = useQueryClient();
    const { toggleSidebar } = useSidebar()

    const saveDashboardMutation = useMutation({
      mutationFn: editDashboard,
      onSuccess: () => {
          toast({
              title: t('Dashboard saved.'),
              variant: 'success',
          });
      },
      onError: (error: AxiosError<{ message: string }>) => {
          toast({
              title: t('Failed to save dashboard.'),
              variant: 'error',
              description: error.response?.data.message,
          });
      },
    });

    const saveReportMutation = useMutation({
      mutationFn: addReport,
      onSuccess: () => {
          queryClient.invalidateQueries({ queryKey: ['reports'] });
          toast({
              title: t('Report saved.'),
              variant: 'success',
          });
      },
      onError: (error: AxiosError<{ message: string }>) => {
          toast({
              title: t('Failed to save report.'),
              variant: 'error',
              description: error.response?.data.message,
          });
      },
    });

    const onClickSaveDashboard = () => {
      saveDashboardMutation.mutate({id: dashboardId, title: dashboardName, user: definition.user, data: JSON.stringify({title: dashboardName, rows: definition.rows, subQueries: subQueries, filters: filters, range: range})})
    }
  
    const onClickSaveReport = () => {
      saveReportMutation.mutate({title: dashboardName, data: JSON.stringify({title: dashboardName, rows: definition.rows, subQueries: subQueries, filters: filters, range: range}), global: false});
    }
  
    return (
      <div className="w-full flex p-[0.6rem] pr-4 justify-between border-b border-border">
        <div className="flex gap-2 items-center justify-center">
          {/* <div onClick={() => toggleFullScreen()} className="absolute m-3 transition cursor-pointer hover:bg-muted rounded-sm">
            {fullScreen ? <SouthEastIcon/> : <NorthWestIcon/>}
          </div> */}
          <Tooltip>
              <TooltipTrigger asChild>
          <Button variant="ghost" size="icon" onClick={toggleSidebar}>
            <PanelLeft className="h-5 w-5" />       
          </Button>
          </TooltipTrigger>
          <TooltipContent>
                {t("Toggle Sidebar")}
              </TooltipContent>
          </Tooltip>
          <div data-orientation="vertical" role="none" className="shrink-0 bg-border w-[1px] h-4"></div>
            <Tooltip>
              <ManageQueriesDialog>
                <TooltipTrigger asChild>
                  <Button variant="ghost" size="icon">
                    <ChatOutlinedIcon fontSize="small" />
                  </Button>
                </TooltipTrigger>
              </ManageQueriesDialog>
              <TooltipContent>
                {t("Manage dashboard queries")}
              </TooltipContent>
            </Tooltip>
            <Tooltip>
              <AddAlertDialog dashboardName={dashboardName} dashboardId={dashboardId}>
                <TooltipTrigger asChild>
                  <Button variant="ghost" size="icon">
                    <NotificationAddOutlinedIcon fontSize="small" />
                  </Button>
                </TooltipTrigger>
              </AddAlertDialog>
              <TooltipContent>
                {t("Create Alert")}
              </TooltipContent>
            </Tooltip>
            <div className="flex bg-accent rounded-md">
              <Tooltip>
                <TooltipTrigger asChild>
                  <Button onClick={onClickSaveDashboard} variant="ghost" size="icon">
                    <SaveOutlinedIcon fontSize="small" />
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  {t("Save")}
                </TooltipContent>
              </Tooltip>
              <SaveAs
                isSaveDashboard={true}
                title={t("Save as")}
                dashboardId={dashboardId}
                definition={definition}
              />
            </div>
            <div className="flex bg-accent rounded-md">
              <Tooltip>
                <TooltipTrigger asChild>
                  <Button onClick={onClickSaveReport} variant="ghost" size="icon">
                    <ArticleOutlinedIcon fontSize="small" />
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  {t("Save Report")}
                </TooltipContent>
              </Tooltip>
              <SaveAs
                isSaveDashboard={false}
                title={t("Save as report")}
                dashboardId={dashboardId}
                definition={definition}
              />
            </div>
            <div className="flex bg-accent rounded-md">
              <ExportDashboardDialog id={dashboardId} dashboard={definition}>
                <Tooltip>
                  <TooltipTrigger>
                    <div className="rounded-md p-1 text-sm hover:cursor-pointer hover:bg-muted">
                      <IosShareIcon fontSize="small" />
                    </div>
                  </TooltipTrigger>
                  <TooltipContent>{t('Export')}</TooltipContent>
                </Tooltip>
              </ExportDashboardDialog>
            </div>
            <div className="flex bg-accent rounded-md">
              <ImportDashboardDialog>
                <Button>{t('Import')}</Button>
              </ImportDashboardDialog>
            </div>
              {/* <Tooltip>
                <TooltipTrigger asChild>
                  <Button variant="ghost" size="icon">
                    <SettingsIcon />
                  </Button>
                </TooltipTrigger>
                <TooltipContent>
                  {t("Configure dashboard")}
                </TooltipContent>
              </Tooltip> */}
          </div>
          <div className="flex items-center gap-2">
            <AddRowDialog>
              <Button variant="secondary" className="gap-2 border border-border">
                <AddIcon fontSize="small" />
                {t("Add Row")}
              </Button>
            </AddRowDialog>
            <ShareDashboardDialog dashboardId={dashboardId}>
              <Button type="submit" variant="secondary" className="gap-2 border border-border">
                <ShareOutlinedIcon fontSize="small" />
                {t("Share")}
              </Button>
            </ShareDashboardDialog>
            <TimePicker />
            <RefreshRateDropdown />
          </div>
      </div>
    )
  
  }