import { useTranslation } from 'react-i18next';
import { Button } from "@/components/shadcn/button";
import AutorenewIcon from '@mui/icons-material/Autorenew';
import { ChevronDown } from "lucide-react";
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuCheckboxItem,
    DropdownMenuSeparator,
    DropdownMenuLabel,
    DropdownMenuTrigger,
} from '@/components/shadcn/dropdown-menu';
import {
    Tooltip,
    TooltipContent,
    TooltipTrigger,
} from '@/components/shadcn/tooltip';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import moment from 'moment';
import { useState } from 'react';
import { cn } from '@/lib/utils';

export const RefreshRateDropdown = () => {
    const { t } = useTranslation();
    const { range, setRange, refreshRate, setRefreshRate } = useDashboardContext().dashboardInfo;
    const [rotate, setRotate] = useState(false);
    const refreshRateOptions = [
        { label: 'Off', value: 0 },
        { label: '10s', value: 10000 },
        { label: '1m', value: 60000 },
        { label: '5m', value: 300000 },
        { label: '15m', value: 900000 },
        { label: '30m', value: 1800000 },
        { label: '1h', value: 3600000 },
        { label: '2h', value: 7200000 },
        { label: '1d', value: 86400000 }
    ];
    const intervalOptions: {[key: string]: { label: string; offset: number }} = {
        '5m': { label: t('Last 5m'), offset: 5 * 60 },
        '15m': { label: t('Last 15m'), offset: 15 * 60 },
        '1h': { label: t('Last 1h'), offset: 60 * 60 },
        '2h': { label: t('Last 2h'), offset: 2 * 60 * 60 },
        '6h': { label: t('Last 6h'), offset: 6 * 60 * 60 },
        '12h': { label: t('Last 12h'), offset: 12 * 60 * 60 },
        '24h': { label: t('Last 24h'), offset: 24 * 60 * 60 },
        '2d': { label: t('Last 2d'), offset: 2 * 24 * 60 * 60 },
        '7d': { label: t('Last 7d'), offset: 7 * 24 * 60 * 60 },
        '30d': { label: t('Last 30d'), offset: 30 * 24 * 60 * 60 }
    };

    const refreshNow = () => {
        if (range.value !== "custom") {
            setRotate(true);
            const end = moment();
            const start = moment(end).subtract(intervalOptions[range.value].offset, 'seconds');
            setRange(
                {
                    ...range,
                    startTime: start,
                    endTime: end,
                    reset: false
                }
            );
        }
    }

    return (
        <div className="rounded-md border border-border flex items-center justify-between min-w-fit bg-secondary">
            <Tooltip>
                <TooltipTrigger>
                    <Button
                        variant="ghost"
                        className="hover:bg-background/20"
                        onClick={() => refreshNow()}
                    >
                        <AutorenewIcon
                            fontSize="small"
                            className={cn(rotate && "animate-[spin_350ms_linear_1]")}
                            onAnimationEnd={() => setRotate(false)}
                        />
                    </Button>
                </TooltipTrigger>
                <TooltipContent>
                    {t("Refresh")}
                </TooltipContent>
            </Tooltip>
            <div data-orientation="vertical" role="none" className="shrink-0 bg-border w-[1px] h-4"></div>
            <DropdownMenu>
                <Tooltip>
                    <TooltipTrigger>
                        <DropdownMenuTrigger asChild>
                            <Button variant="ghost" className="hover:bg-background/20">
                                <ChevronDown className="h-4 w-4" />
                            </Button>
                        </DropdownMenuTrigger>
                    </TooltipTrigger>
                    <TooltipContent>
                        {t("Auto refresh")}
                    </TooltipContent>
                </Tooltip>
                <DropdownMenuContent>
                    <DropdownMenuLabel className="text-center">
                        {t("Auto Refresh")}
                    </DropdownMenuLabel>
                    <DropdownMenuSeparator />
                    {refreshRateOptions.map((option) => (
                        <DropdownMenuCheckboxItem
                            checked={refreshRate === option.value}
                            onClick={() => setRefreshRate(option.value)}
                        >
                            {option.label}
                        </DropdownMenuCheckboxItem>
                    ))}
                </DropdownMenuContent>
            </DropdownMenu>
        </div>
    )

}