import { Popover, PopoverContent, PopoverTrigger } from '@/components/shadcn/popover';
import {
    Tooltip,
    TooltipContent,
    TooltipTrigger,
} from "@/components/shadcn/tooltip";
import { Button } from '@/components/shadcn/button';
import { Input } from '@/components/shadcn/input';
import { useState } from 'react';
import { DashboardDefinition } from '@/lib/types';
import ArrowDropDownIcon from '@mui/icons-material/ArrowDropDown';
import ArrowRightIcon from '@mui/icons-material/ArrowRight';
import PublicIcon from '@mui/icons-material/Public';
import { useTranslation } from 'react-i18next';
import { useDashboardContext } from '@/dashboard/contexts/DashboardContext';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { toast } from '@/components/shadcn/use-toast';
import { AxiosError } from 'axios';
import { addDashboard } from '@/dashboard/queries';
import { addReport } from '@/dashboard/reports/queries';

export interface SaveAsProps {
    isSaveDashboard: boolean,
    title: string,
    dashboardId: string,
    definition: DashboardDefinition,
}

export const SaveAs = (props : SaveAsProps) => {
    const [newName, setNewName] = useState("");
    const { t } = useTranslation();
    const { range, subQueries, filters, refreshRate } = useDashboardContext().dashboardInfo;
    const queryClient = useQueryClient();
    const [open, setOpen] = useState(false);

    const addDashboardMutation = useMutation({
        mutationFn: addDashboard,
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['dashboards'] });
            setOpen(false);
            toast({
                title: t('Saved new Dashboard.'),
                variant: 'success',
                description: t('Select your new dashboard in the dashboard dropdown menu.'),
            });
        },
        onError: (error: AxiosError<{ message: string }>) => {
            toast({
                title: t('Failed to Add Dashboard.'),
                variant: 'error',
                description: error.response?.data.message || t('An unknown error occured.'),
            });
        },
    });

    const saveReportMutation = useMutation({
        mutationFn: addReport,
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['reports'] });
            toast({
                title: t('Report saved.'),
                variant: 'success',
            });
        },
        onError: (error: AxiosError<{ message: string }>) => {
            toast({
                title: t('Failed to save report.'),
                variant: 'error',
                description: error.response?.data.message,
            });
        },
      });

    // const onExportDashboard = () => {
    //     // TODO: Implement this
    //     return;
    // }

    const onClickSaveDashboardAs = (global: boolean) => {
        addDashboardMutation.mutate({title: newName, data: JSON.stringify({title: newName, rows: props.definition.rows, subQueries: subQueries, filters: filters, range: range, refreshRate: refreshRate}), global: global});
        return;
    }

    const onClickSaveReportAs = (global: boolean) => {
        saveReportMutation.mutate({title: newName, data: JSON.stringify({title: newName, rows: props.definition.rows, subQueries: subQueries, filters: filters, range: range}), global: global});
        return;
    }

    const onSaveAs = (props.isSaveDashboard) ? onClickSaveDashboardAs : onClickSaveReportAs;

    return (
        <Popover open={open} onOpenChange={setOpen}>
            <PopoverTrigger>
                <Tooltip>
                    <TooltipTrigger asChild>
                        <Button variant="ghost" size="icon">
                            <ArrowDropDownIcon />
                        </Button>
                    </TooltipTrigger>
                    <TooltipContent>
                        {props.title}
                    </TooltipContent>
                </Tooltip>
            </PopoverTrigger>
            <PopoverContent className="w-fit">
                <div className="flex w-full max-w-sm items-center space-x-2 flex-nowrap">
                    <Input
                        type="text"
                        placeholder="Save as..."
                        className="flex-grow p-2 border-border"
                        value={newName}
                        onChange={(e) => setNewName(e.target.value)}
                    />
                    <Tooltip>
                        <TooltipTrigger asChild>
                            <Button onClick={() => onSaveAs(false)} className="px-2">
                                <ArrowRightIcon />
                            </Button>
                        </TooltipTrigger>
                        <TooltipContent>
                            {t("Save as ") + ((props.isSaveDashboard) ? t("dashboard") : t("report")) + "..."}
                        </TooltipContent>
                    </Tooltip>
                    <Tooltip>
                        <TooltipTrigger asChild>
                            <Button onClick={() => onSaveAs(true)} className="px-2">
                                <PublicIcon />
                                <ArrowRightIcon />
                            </Button>
                        </TooltipTrigger>
                        <TooltipContent>
                            {t("Save as global ") + ((props.isSaveDashboard) ? t("dashboard") : t("report")) + "..."}
                        </TooltipContent>
                    </Tooltip>
                </div>
            </PopoverContent>
        </Popover>
    )
}