import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuSub,
    DropdownMenuSubContent,
    DropdownMenuSubTrigger,
    DropdownMenuTrigger,
} from "@/components/shadcn/dropdown-menu";
import moment from 'moment';
import * as React from "react";
import { format } from "date-fns";
import { DateRange } from "react-day-picker";
import { cn } from "@/lib/utils";
import { Button } from "@/components/shadcn/button";
import { Label } from "@/components/shadcn/label";
import { Calendar } from "@/components/shadcn/calendar";
import { ChevronDown } from "lucide-react";
import { Calendar as CalendarIcon } from "lucide-react";
import { useTranslation } from 'react-i18next';
import { Range } from "@/lib/types";
import { useDashboardContext, getLocaleTimeFormat } from '@/dashboard/contexts/DashboardContext';
import { Input } from "@/components/shadcn/input";

const DatePickerWithRange = ({ range, setRange }: { range: Range, setRange: (range: Range) => void} ) => {
  const { t } = useTranslation();
  const dateFormat = getLocaleTimeFormat(['year', 'day'], {month: 'short'});
  const dateTimeFormat = getLocaleTimeFormat(['month', 'minute'], {month: 'short'});

  const [date, setDate] = React.useState<DateRange | undefined>({
    from: range.startTime.toDate(),
    to: range.endTime.toDate(),
  })

  const makeInterval = (start: Date | undefined, end: Date | undefined) => {
    if (start === undefined && end === undefined) { return t("Pick a date"); }
    else if (end === undefined) { return dateTimeFormat.format(start); }
    
    let interval:string;
    if (start?.getFullYear() != end.getFullYear()) { 
      interval = dateFormat.format(start) + " - " + dateFormat.format(end);
    }
    else { 
      interval = dateTimeFormat.format(start) + " - " + dateTimeFormat.format(end);
    }
    return interval;
  }

  const handleRangeChange = (newDateRange: DateRange | undefined) => {
    if (newDateRange != undefined) {
      const start = moment(newDateRange.from)
                    .hour(date?.from? date.from.getHours() : 0)
                    .minute(date?.from? date.from.getMinutes() : 0)
                    .second(date?.from? date.from.getSeconds() : 0).millisecond(0);
      const end = moment(newDateRange.to)
                  .hour(date?.to? date.to.getHours() : 23)
                  .minute(date?.to? date.to.getMinutes() : 59)
                  .second(date?.to? date.to.getSeconds() : 59).millisecond(999);
      if (start.isAfter(end)) {
        end.hour(start.hour()).add(1, 'hour') 
      }
      // if selecting same day, and time is also the same
      if (start.year() == end.year() && 
          dateTimeFormat.format(start.toDate()) == dateTimeFormat.format(end.toDate()) && 
          start.second() == end.second()) {
        end.hour(23).minute(59).second(59)
      }

      const interval: string = makeInterval(start.toDate(), end.toDate());

      setDate({
        from: start.toDate(),
        to: end.toDate()
      });

      setRange({
        startTime: start,
        endTime: end,
        value: "custom",
        label: interval,
        reset: true
      });
    }
  }

  const handleTimeChange = (newTime: string | null, from: "start" | "end") => {
    console.log(newTime);
    if (newTime == null || date == null) { return; }
    const timeSplit = newTime.split(':').map(Number);
    const start = date? moment(date.from) : moment();
    const end = date ? moment(date.to) : moment();
    if (from == "start") {
      start.hour(timeSplit[0]).minute(timeSplit[1]).second(timeSplit[2]).millisecond(0);
      if (start.isAfter(end)) {
        if (end.hour() < start.hour()) { 
          end.hour(start.hour()).add(1, 'hour') 
        }
        else if (end.minute() < start.minute()) { 
          end.minute(start.minute()).add(1, 'minute') 
        }
        else {
          end.second(start.second()).add(1, 'second') 
        }
      }
    }
    else {
      end.hour(timeSplit[0]).minute(timeSplit[1]).second(timeSplit[2]).millisecond(999);
      if (end.isBefore(start)) {
        if (start.hour() > end.hour()) { 
          start.hour(end.hour()).subtract(1, 'hour') 
        }
        else if (start.minute() > end.minute()) { 
          start.minute(end.minute()).subtract(1, 'minute') 
        }
        else { 
          start.second(end.second()).subtract(1, 'second') 
        }
      }
    }

    const interval = makeInterval(start.toDate(), end.toDate());

    setDate({
      from: start.toDate(),
      to: end.toDate()
    })

    setRange({
        startTime: start,
        endTime: end,
        value: "custom",
        label: interval,
        reset: true
      })
  }

  return (
    <div className={cn("grid gap-2")}>
      <DropdownMenuSub>
        <DropdownMenuSubTrigger>
          {t("Custom")}
        </DropdownMenuSubTrigger>
        <DropdownMenuSubContent className="w-auto p-0">
          <div className="flex justify-center">
          <Calendar
            showYearSwitcher
            autoFocus
            mode="range"
            defaultMonth={date?.from}
            selected={date}
            onSelect={(date) => {handleRangeChange(date)}}
            numberOfMonths={1}
          />
          </div>
          <div className="flex justify-evenly p-px">
            <div className="p-px">
            <Label htmlFor="start-time-select" className="flex justify-center">{t("Start")}</Label>
            <Input type="time" id="start-time-select" step="1"
                   value={date ? date.from ? format(date.from, "HH:mm:ss") : range.startTime.format("HH:mm:ss") : range.startTime.format("HH:mm:ss")}
                   onChange={(time) => {handleTimeChange(time.currentTarget.value, 'start')}}/>
            </div>
            <div className="p-px">
            <Label htmlFor="end-time-select" className="flex justify-center">{t("End")}</Label>
            <Input type="time" id="end-time-select"  step="1"
                   value={date? date.to ? format(date.to, "HH:mm:ss") : range.startTime.format("HH:mm:ss") : range.startTime.format("HH:mm:ss")}
                   onChange={(time) => {handleTimeChange(time.currentTarget.value, 'end')}}/>
            </div>
          </div>
        </DropdownMenuSubContent>
      </DropdownMenuSub>
    </div>
  )
}

export function TimePicker() {
    const { t } = useTranslation();
    const { range, setRange } = useDashboardContext().dashboardInfo;

    const intervalOptions: {[key: string]: { label: string; offset: number }} = {
      '5m': { label: t('Last 5m'), offset: 5 * 60 },
      '15m': { label: t('Last 15m'), offset: 15 * 60 },
      '1h': { label: t('Last 1h'), offset: 60 * 60 },
      '2h': { label: t('Last 2h'), offset: 2 * 60 * 60 },
      '6h': { label: t('Last 6h'), offset: 6 * 60 * 60 },
      '12h': { label: t('Last 12h'), offset: 12 * 60 * 60 },
      '24h': { label: t('Last 24h'), offset: 24 * 60 * 60 },
      '2d': { label: t('Last 2d'), offset: 2 * 24 * 60 * 60 },
      '7d': { label: t('Last 7d'), offset: 7 * 24 * 60 * 60 },
      '30d': { label: t('Last 30d'), offset: 30 * 24 * 60 * 60 }
    };
  
    const handleIntervalChange = (newInterval: string) => {
      const now = moment();
      const intervalValue = intervalOptions[newInterval];
      const start = moment(now).subtract(intervalValue.offset, 'seconds');
      setRange({
        startTime: start,
        endTime: now,
        value: newInterval,
        label: intervalValue.label,
        reset: true
      });
    }

    return (
      <div>
        <DropdownMenu>
          <DropdownMenuTrigger asChild>
            <Button variant="secondary" className="border border-border flex items-center justify-between min-w-fit gap-2">
                <div className="flex items-center">
                  <CalendarIcon className="mr-2 h-4 w-4" />
                  {range.label}
                </div>
                <ChevronDown className="h-4 w-4" />
            </Button>
          </DropdownMenuTrigger>
          <DropdownMenuContent>
            {Object.keys(intervalOptions).map((option) => (
              <DropdownMenuItem key={option} onClick={() => handleIntervalChange(option)}>
                {intervalOptions[option].label}
              </DropdownMenuItem>
            ))}
            <DatePickerWithRange range={range} setRange={setRange} />
          </DropdownMenuContent>
        </DropdownMenu>
      </div>
    );

}