<?php

/**
 * @brief Validates the given NRDP server and token combination
 * 
 * @param $server The URL of the NRDP server
 * @param $token The token to access the NRDP server
 * 
 * Tests whether or not the given NRDP server/token combo is legitimate.
 * Attempts to connect to the NRDP server, and on failure throws
 * and acceptable reason for failure.
 */
function validate_nrdp_server_with_token($server, $token, $allow_self_sign)
{
    $url = "{$server}?token={$token}&cmd=submitcmd";
    $verify_peer = ($allow_self_sign == 'true') ? false : true;
    $curl_handle = curl_init($url);
    curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($curl_handle, CURLOPT_SSL_VERIFYHOST, $verify_peer);
    curl_setopt($curl_handle, CURLOPT_SSL_VERIFYPEER, $verify_peer);
    $resp = curl_exec($curl_handle);
    $code = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);

    if ($xml === false || $code != 200) {
        $message = curl_error($curl_handle);
    }

    curl_close($curl_handle);
    $output['message'] = $message;
    $output['code'] = $code;
    return $output;
}

function get_alert_seconds($period)
{
    $suffix = substr($period, -1);
    $num = intval(substr($period, 0, strlen($period)-1));
    if ($num == 0) {
        $num = 1;
    }

    $base = 60;
    switch ($suffix)
    {
        case 'M':
        case 'm':
            $base = 60; 
            break;

        case 'H':
        case 'h':
            $base = 60*60;
            break;

        case 's':
        case 'S':
            $base = 1;
            break;

        case 'd':
        case 'D':
            $base = 24*60*60;
            break;

        case 'w':
        case 'W':
            $base = 7*24*60*60;
            break;
    }

    // Do calculation
    $seconds = $base * $num;
    return $seconds;
}

function get_alert_status($code)
{
    switch ($code)
    {
        case 0:
            $status = "ok";
            break;

        case 1:
            $status = "warning";
            break;

        case 2:
            $status = "critical";
            break;

        default:
            $status = "unknown";
            break;
    }

    return $status;
}

function reschedule_alert($last_run, $check_interval)
{
    $seconds = get_alert_seconds($check_interval);
    $next_run = $last_run + $seconds;
    return $next_run;
}

/**
 * Verify that the alert needs to be ran... this happens because it's possible
 * the alert was ran by a different node before this one started it.
 **/
function verify_alert_run($alert_id)
{
    $ci =& get_instance();
    $alert = $ci->elasticsearch->get('alert', $alert_id);
    if ($alert['_source']['next_run'] <= time() && $alert['_source']['active'] == 1) {
        return true;
    }
    return false;
}

// Get the default email message for alerts
function get_default_email_tpl($force=false)
{
    $ci =& get_instance();
    $tpl = array();
    $cf_default = get_option('default_email_tpl', 'system');

    // Grab the config option, if it exists and fill out the template info
    if ($cf_default == 'system' || $force) {
        $tpl['name'] = _('System Default');
        $tpl['subject'] = _('Check returned') . ' %state% ';
        $tpl['body'] = '<p>%alertname% '._('returned with a').' <b>%state%</b> '._('state at').' <b>%time%</b></p>

<p>'._('The alert was processed with the following thresholds').':<br>
<ul>
    <li>'._('Lookback period').': %lookback%</li>
    <li>'._('Warning').': %warning%</li>
    <li>'._('Critical').': %critical%</li>
</ul>
</p>

<p>
'._('Here is the full alert output').':
<div style="padding: 10px; background-color: #F9F9F9;">%output%</div>
</p>

<p>
Non-Sending Hosts:
<div style="padding: 10px; background-color: #F9F9F9;">%hostnames%</div>
</p>

<p>'._('See the last').' %lookback% '._('in the').' <a href="%url%">'._('Nagios Log Server dashboard').'</a>.</p>

<p>'._('Nagios Log Server').'</p>';

        if (get_option('email_text_only', 0))  {
            $tpl['body'] = strip_tags($tpl['body']);
        }

    } else {
        // Look up email template info
        $template = $ci->elasticsearch->get('email_template', $cf_default);
        if (!empty($template['found'])) {
            $tpl['name'] = $template['_source']['name'];
            $tpl['subject'] = $template['_source']['subject'];

            $body = $template['_source']['body'];
            if (base64_decode($body, true) === false) {
                $tpl['body'] = $body;
            } else {
                $tpl['body'] = base64_decode($body);
            }
        }
    }

    return $tpl;
}