<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
 * CodeIgniter Session OpenSearch Driver
 *
 */

class CI_Session_opensearch_driver extends CI_Session_driver implements SessionHandlerInterface
{
	/**
	 * Validate SID regular expression
	 *
	 * @var	string
	 */
	protected $_sid_regexp;

	/**
	 * mbstring.func_overload flag
	 *
	 * @var	bool
	 */
	protected static $func_overload;

	/**
	 * OpenSearch connection instance
	 *
	 * @var	object
	 */
    protected $_opensearch = null;

	/**
	 * session timestamp
	 *
	 * @var	number
	 */
    protected $_timestamp = 0;

	// ------------------------------------------------------------------------

	/**
	 * Class constructor
	 *
	 * @param	array	$params	Configuration parameters
	 * @return	void
	 */
	public function __construct(&$params)
	{
		parent::__construct($params);

        $this->_opensearch = new Elasticsearch([ 'index' =>  'nagioslogserver_sessions' ]);
        $this->_timestamp = microtime(true);
        $this->_sid_regexp = $this->_config['_sid_regexp'];

		isset(self::$func_overload) OR self::$func_overload = (extension_loaded('mbstring') && ini_get('mbstring.func_overload'));
	}

	// ------------------------------------------------------------------------

	/**
	 * Open
	 *
	 * Sanitizes the save_path directory.
	 *
	 * @param	string	$save_path	Path to session files' directory
	 * @param	string	$name		Session cookie name
	 * @return	bool
	 */
	public function open($save_path, $name)
	{
        return $this->_success;
	}

	// ------------------------------------------------------------------------

	/**
	 * Read
	 *
	 * Reads session data and acquires a lock
	 *
	 * @param	string	$session_id	Session ID
	 * @return	string	Serialized session data
	 */
	public function read($session_id)
	{
        if ($this->_opensearch === NULL) {
            // Initialize storage mechanism (connection)
            $this->_opensearch = new Elasticsearch([ 'index' =>  'nagioslogserver_sessions' ]);
            if ($this->_opensearch === null) {
                return $this->_failure;
            }
        }

        // Needed by write() to detect session_regenerate_id() calls
		$this->_session_id = $session_id;

        // Read session data (if exists), acquire locks
        $session = $this->_opensearch->get('',$session_id);
        if ($session === null || empty($session['found'])) {
            $retVal = $this->_failure;
        } else {
            $retVal = $session['_source']['data']['data'];
        }

        if ($retVal !== $this->_failure) {
            $this->_fingerprint = md5($retVal);
            return $retVal;
        }

		$session_data = '';

		$this->_fingerprint = md5($session_data);
		return $session_data;
	}

	// ------------------------------------------------------------------------

	/**
	 * Write
	 *
	 * Writes (create / update) session data
	 *
	 * @param	string	$session_id	Session ID
	 * @param	string	$session_data	Serialized session data
	 * @return	bool
	 */
	public function write($session_id, $session_data)
	{

        // If the two IDs don't match, we have a session_regenerate_id() call
		// and we need to close the old handle and open a new one
		if ($session_id !== $this->_session_id && ($this->close() === $this->_failure OR $this->read($session_id) === $this->_failure))
		{
            return $this->_failure;
		}

        if ($this->_fingerprint === md5($session_data))
		{
            return $this->_success;
		}

        // Ensure that the data has a user_id in it before writing
        if (strpos($session_data, 'user_id') === false) {
            return $this->_success;
        }

        $existing_session = $this->_opensearch->get('', $session_id);
        if ($existing_session === null || empty($existing_session['found'])) {
            $this->_opensearch->add('', [ 'data' => [ 'data' => $session_data, 'last_activity' => microtime(true) ] ], $session_id);
        } else {
            $this->_opensearch->update('', [ 'data' => [ 'data' => $session_data, 'last_activity' => microtime(true) ] ], $session_id);
        }

		$this->_fingerprint = md5($session_data);
		return $this->_success;
	}

	// ------------------------------------------------------------------------

	/**
	 * Close
	 *
	 * Releases locks and closes file descriptor.
	 *
	 * @return	bool
	 */
	public function close()
	{
        $this->_session_id = NULL;
        return $this->_success;
	}

	// ------------------------------------------------------------------------

	/**
	 * Destroy
	 *
	 * Destroys the current session.
	 *
	 * @param	string	$session_id	Session ID
	 * @return	bool
	 */
	public function destroy($session_id)
	{
        $this->_cookie_destroy();
        $result = $this->_opensearch->delete('', $this->_session_id);
        $result = $this->_opensearch->delete('', $session_id);
        $this->_session_id = null;
        return $this->_success;
	}

	// ------------------------------------------------------------------------

	/**
	 * Garbage Collector
	 *
	 * Deletes expired sessions
	 *
	 * @param	int 	$maxlifetime	Maximum lifetime of sessions
	 * @return	bool
	 */
	public function gc($maxlifetime)
	{
        $ts = time() - $maxlifetime;

        $delete_query = [
            'query' => [
                'range' => [
                    'data.last_activity' => [
                        'lt' => $ts
                    ]
                ]
            ]
        ];

        $this->_opensearch->delete_by_query($delete_query);

		return $this->_success;
	}

	// --------------------------------------------------------------------

	/**
	 * Byte-safe strlen()
	 *
	 * @param	string	$str
	 * @return	int
	 */
	protected static function strlen($str)
	{
		return (self::$func_overload)
			? mb_strlen($str, '8bit')
			: strlen($str);
	}
}
