/*
NOTE:
This file uses puppeteer which requires node.
In order to avoid installing node and puppeteer on users' machines, it is bundled into a single executable.
It is first bundled with puppeteer using esbuild (https://esbuild.github.io) then bundled with node using SEA (https://nodejs.org/api/single-executable-applications.html).
Whenever you make changes, run the following (things may have changed, you should be able to figure it out).
Prerequisites:
- Install node (https://nodejs.org/en/download)
Esbuild:
- Install esbuild: npm install --save-exact --save-dev esbuild
- Check it's installed: ./node_modules/.bin/esbuild --version
- Bundle the file: ./node_modules/.bin/esbuild export.js --bundle --outfile=out.js --platform=node --external:typescript
SEA:
- Create config file: echo '{ "main": "out.js", "output": "sea-prep.blob" }' > sea-config.json
- Generate blob: node --experimental-sea-config sea-config.json
- Copy the node executable: cp $(command -v node) out
- Inject the blob:
  npx postject out NODE_SEA_BLOB sea-prep.blob \
    --sentinel-fuse NODE_SEA_FUSE_fce680ab2cc467b6e072b8b5df1996b2
Postrequisites:
  - Make sure the bundled file is in /repos/nagioslogserver/nagioslogserver/basedir/html/www/media/js/
  - Rename the file to export
*/

import puppeteer from 'puppeteer';
// Or import puppeteer from 'puppeteer-core';

async function main() {
  const args = process.argv;
  const token = args[2];
  const type = args[3];
  const path = args[4];
  const id = args[5];

  try {
    // Launch the browser and open a new blank page
    const browser = await puppeteer.launch({
      args: ['--no-sandbox', '--disable-setuid-sandbox'],  // Add these flags to fix the root sandbox issue
      executablePath: '/usr/bin/chromium-browser'
    });

    // Fetch the token data
    const response = await fetch(`http://localhost/nagioslogserver/reports/shmeap?token=${token}`);
    const cookies = response.headers.get('set-cookie');
    const cookieArray = cookies.split(',');
    const lsSessionCookie = cookieArray.find(cookie => cookie.trim().startsWith('ls_session='));
    const lsSessionValue = lsSessionCookie.split('=')[1].split(';')[0];

    // Set the session cookie for Puppeteer
    await browser.setCookie({
      name: 'ls_session',
      value: lsSessionValue,
      domain: 'localhost',
      path: '/',
      httpOnly: true,
      secure: false
    });

    // Open a new page
    const page = await browser.newPage();

    if (type === 'jpg') { await page.setViewport({ width: 1480, height: 720 }); }

    // Set the URL for the page based on input
    const url = `http://localhost/nagioslogserver/media/ui/index.html?dashboardId=${id}&report=true&exporting=true`;
    
    // Navigate to the URL
    await page.goto(url, {
      waitUntil: 'networkidle0'
    });

    // Depending on the type, either save a PDF or screenshot
    if (type === 'pdf') {
      await page.pdf({
        path: path,
        format: 'A4',
        landscape: true,
        scale: 0.75,
        printBackground: true
      });
    } else if (type === 'jpg') {
      await page.screenshot({
        path: path,
        fullPage: true,
        type: 'jpeg'
      });
    }

    // Close the browser
    await browser.close();
  } catch (error) {
    console.error('Error occurred:', error);
  }
}

// Call the main function
main();
