#!/bin/bash

# Function to validate the provided API key
check_api_key() {
    if [ -z "$API_KEY" ]; then
        echo "API key not provided. Use -k option to provide it."
        echo "Exiting with status 1"
        exit 1
    fi
}

# Function to validate the provided address and port
check_self_hosted() {
    if [ -z "$ADDRESS" ] || [ -z "$PORT" ]; then
        echo "Address or port not provided. Use --address and --port options to provide them."
        echo "Exiting with status 1"
        exit 1
    fi
}

# Function to process command line arguments
process_args() {
    while [[ "$#" -gt 0 ]]; do
        case $1 in
            -k|--key)
                API_KEY="$2"
                shift
                ;;
            --local)
                LOCAL="true"
                ;;
            --address)
                ADDRESS="$2"
                shift
                ;;
            --port)
                PORT="$2"
                shift
                ;;
            *)
                echo "Invalid option: $1"
                echo "Usage: $0 -k '<api_key_value>' [--local --address <address> --port <port>]"
                echo "Exiting with status 1"
                exit 1
                ;;
        esac
        shift
    done
}

# Function to test API key via curl with a timeout
test_api_key_with_timeout() {
    local RESPONSE=$(curl -s --max-time 3 "$@")
    if [ $? -ne 0 ]; then
        echo "Timeout, the API provider may be having issues."
        exit 1
    fi
    echo "$RESPONSE"
}

# Function to test API key via curl for OpenAI
test_openai_api_key() {
    local RESPONSE=$(test_api_key_with_timeout https://api.openai.com/v1/chat/completions \
        -H 'Content-Type: application/json' \
        -H "Authorization: Bearer $API_KEY" \
        -d '{
        "model": "gpt-3.5-turbo",
        "messages": [
            {
                "role": "system",
                "content": "You are a helpful assistant."
            },
            {
                "role": "user",
                "content": "Hello!"
            }
        ],
        "max_tokens": 1,
        "temperature": 1.0
      }')

    if [[ "$RESPONSE" == *'"error":'* || "$RESPONSE" == *'"message": "Incorrect API key provided:'* ]]; then
        OPENAI_VALID="no"
    else
        OPENAI_VALID="yes"
    fi
}

# Function to test API key via curl for Mistral
test_mistral_api_key() {
    local RESPONSE=$(test_api_key_with_timeout -s --location "https://api.mistral.ai/v1/chat/completions" \
        -H 'Content-Type: application/json' \
        -H 'Accept: application/json' \
        -H "Authorization: Bearer $API_KEY" \
        -d '{
        "model": "mistral-small-latest",
        "messages": [{"role": "user", "content": "hi?"}],
        "max_tokens": 1
      }')

    if [[ "$RESPONSE" == *"Unauthorized"* ]]; then
        MISTRAL_VALID="no"
    else
        MISTRAL_VALID="yes"
    fi
}

# Function to test API key via curl for Anthropic
test_anthropic_api_key() {
    local RESPONSE=$(test_api_key_with_timeout -s https://api.anthropic.com/v1/complete \
        -H 'content-type: application/json' \
        -H "anthropic-version: 2023-06-01" \
        -H "x-api-key: $API_KEY" \
        -d '{
          "model": "claude-2",
          "prompt": "just say hi",
          "max_tokens_to_sample": 1,
          "stream": true
        }')

    if [[ "$RESPONSE" == *'"error":'* && "$RESPONSE" == *'invalid x-api-key'* ]]; then
        ANTHROPIC_VALID="no"
    else
        ANTHROPIC_VALID="yes"
    fi
}

# Function to test self-hosted IP and port
test_self_hosted() {
    self_hosting_available=true  

    if ! $self_hosting_available; then
        echo "Error: Self-hosting is unavailable."
        exit 1
    fi

     if [[ "$ADDRESS" != http* ]]; then
        echo "Invalid self-hosted IP and port"
        exit 1
    fi

    local RESPONSE=$(curl -s --max-time 3 $ADDRESS:$PORT/v1/chat/completions \
        -H "Content-Type: application/json" \
        -d '{
            "model": "models/llama-3-lucene-8b",
            "messages": [{"role": "user", "content": "Say this is a test!"}],
            "temperature": 0.7
        }')

    if [[ "$RESPONSE" == *'"role":"assistant"'* ]]; then
        echo "Valid self-hosted IP and port"
        exit 0
    else
        echo "Invalid self-hosted IP and port"
        exit 1
    fi
}

# Main execution
process_args "$@"

if [ "$LOCAL" == "true" ]; then
    check_self_hosted
    test_self_hosted
fi

check_api_key

OPENAI_VALID="no"
MISTRAL_VALID="no"
ANTHROPIC_VALID="no"

test_openai_api_key
test_mistral_api_key
test_anthropic_api_key

echo "OpenAI API key is valid: $OPENAI_VALID"
echo "Mistral API key is valid: $MISTRAL_VALID"
echo "Anthropic API key is valid: $ANTHROPIC_VALID"

if [[ "$OPENAI_VALID" == "yes" ]]; then
    echo "Exiting with status 10 (OpenAI valid)"
    exit 10
elif [[ "$MISTRAL_VALID" == "yes" ]]; then
    echo "Exiting with status 20 (Mistral valid)"
    exit 20
elif [[ "$ANTHROPIC_VALID" == "yes" ]]; then
    echo "Exiting with status 30 (Anthropic valid)"
    exit 30
else
    echo "Exiting with status 1 (All APIs invalid)"
    exit 1
fi
