#
# libinstall.sh - common functions for installing Nagios software
#

# To use, simply put this file into the same directory as your install file.
# Then add this to the top of your install script:
#     cd $(dirname $(readlink -e "$0"))
#     . ./libinstall.sh
# (The cd command ensures your script is run from the install directory so the
# library can be found, and readlink ensures this works even with symlinks.)
# This library will then automatically run some standard checks for you like
# making sure it's run as root, making sure RHEL systems are registered, etc.
# It will also detect various information about the OS and make that available
# in global variables (see the set_os_info function for the list of variables).
#
# Write your installation steps either as functions or separate scripts. Then
# create a fullinstall function that puts it all together by passing those
# steps to the run_steps wrapper function, which will take care of some error
# handling and progress output. Finally, pass your fullinstall function to the
# log_it wrapper function, which will log the output of the installer to the
# specified file.


# Exit immediately if an untested command fails
set -e

# Global environment variable that determines whether to prompt the user for various information or not
export INTERACTIVE="True"

# Wrapper function for running installation steps
run_steps() {
	local traps
	traps=$(trap)

	for step; do
		# Set trap to print error msg if step fails
		trap "step_error $step" 0

		echo "Running '$step'..."

		if [ -f "installed.$step" ] || [ -f "installed.all" ]; then
			echo "$step step already completed - skipping"

			# Reset traps
			eval "$traps"

			continue
		fi

		"$step"

		echo "$step step completed OK"
		touch "installed.$step"
	done

	# Reset traps
	eval "$traps"

	touch installed.all
	for step; do
		rm -f "installed.$step"
	done
}

# Gets run automatically when an installation step fails
step_error() {
	cat >&2 <<-EOF

		===================
		INSTALLATION ERROR!
		===================
		Installation step failed - exiting.
		Check for error messages in the install log (install.log).

		If you require assistance in resolving the issue, please include install.log
		in your communications with Nagios Enterprises technical support.

		The step that failed was: '$1'
	EOF
}

# Wrapper function for capturing and logging stdout & stderr of another command
# First argument is the log file, followed by the command to run
log_it() {
	log="$1"
	shift
	"$@" 2>&1 | tee -a "$log"
}

# Print installation header text. Takes one argument - the name of the product.
print_header() {
	local product
	product="$1"

	cat <<-EOF

		$product Installation
		$(echo $product Installation | sed 's/./=/g')
		DATE: $(date)
	
		DISTRO INFO:
		$distro
		$version
		$architecture

	EOF
}

# Print installation footer text. Takes two arguments - the name of the product
# and optionally the web server path to the web interface.
print_footer() {
	local product path
	product="$1"
	path="$2"
	ADD_NODE="$3"

	echo ""
	echo "$product Installation Success!"
	echo ""

	get_ip
	MESSAGE="You can finish the final setup steps for $product by visiting:"
	if [ "${ADD_NODE}" = "true" ] ; then
		MESSAGE="You can now access $product by visiting:"
	fi
	
	if [ -n "$path" ]; then
		echo $MESSAGE
		echo "    http://$ip/$path/"
		echo ""
	fi
}

# Get the IP address of the specified device (or eth0 if none specified)
get_ip() {
    if [ "$dist" == "el8" ] || [ "$dist" == "el9" ] || [ "$distro" == "Ubuntu" ] || [ "$distro" == "Debian" ]; then
		ip=$(ip addr | grep global | grep -m 1 'inet' | awk '/inet[^6]/{print substr($2,0)}' | sed 's|/.*||')
	else
		ip=$(ifconfig | egrep -1 eth[0-9] | grep -m 1 'inet' | awk '/inet[^6]/{print substr($2,6)}')
	fi
	if [ "$ip" == "" ];then
		ip="<HOSTNAME>"
	fi
}

# Convenience function for printing errors and exiting
error() {
	echo "ERROR:" "$@" >&2
	return 1
}

# Return successfully only if the specified RPM packages are installed
is_installed() {
	for pkg; do
		rpm -q "$pkg" &>/dev/null
	done
}

# Adds the specified repo to the system package manager. Can be one of:
#     epel, cr (centos' continuous release repo)
add_yum_repo() {
	local repo url pkg
	repo="$1"

	# See if we need to install the repo...
	if is_installed "$repo-release"; then
		echo "$repo-release RPM installed OK"
		return 0
	fi

	echo "Enabling $repo repo..."

	case "$repo" in
		epel )
			if ([ "$dist" = "el8" ] || [ "$dist" = "el9" ]) && [ "$distro" == "CentOS" ]; then
				yum install -y epel-release
			else
				pkg="epel-release-latest-$ver.noarch.rpm"
				url="https://dl.fedoraproject.org/pub/epel/$pkg"
			fi
			;;
		cr )
			if [ "$dist" = "el6" ] && is_installed centos-release; then
				yum -y install centos-release-cr
			fi
	esac

	if [ -n "$url" ] && [ -n "$pkg" ]; then
		curl -L -O "$url"
		rpm -Uvh "$pkg"
		rm "$pkg"
	fi

	yum check-update || true

	# Check to make sure RPM was installed
	if is_installed "$repo-release"; then
		echo "$repo-release RPM installed OK"
	else
		error "$repo-release RPM was not installed - exiting."
	fi
}

# Adds specified user if it doesn't exist already
add_user() {
	local user
	user="$1"

	if ! grep -q "^$user:" /etc/passwd; then
		case "$dist" in
			el* )
				useradd -n "$user"
				;;
			* )
				useradd "$user"
		esac
	fi

	# Add user home folder if it doesn't already exist
	if [ ! -d "/home/$user" ]; then
		mkdir "/home/$user"
		chown $user:$user "/home/$user"
	fi
}

# Adds specified group if it doesn't exist already
add_group() {
	local group
	group="$1"

	if ! grep -q "^$group:" /etc/group; then
		groupadd "$group"
	fi
}

# Adds user to the specified groups
add_to_groups() {
	local user
	user="$1"

	shift
	for group; do
		usermod -a -G "$group" "$user"
	done
}

# Sets the users' primary (default) group if it already exists.
# By default, files/directories created by this user will be created as <user>:<user's primary group>.
set_primary_group() {
	local user
	local group
	user="$1"
	group="$2"

	if grep -q "^$group:" /etc/group; then
		usermod -g "$group" "$user"
	fi
}

# Prompt the user for a custom MySQL password and sets the mysqlpass variable
get_cluster_info() {
	
	if [ "$ADDING_NODE" = "true" ];then
	
		if [ "x$INTERACTIVE" = "xFalse" -a "x$MANAGER_BASE_URL" = "x" ];then
			error "You must specify another cluster hostname with the -m flag in non-interactive mode"
		fi
		
		echo ""
		echo ""
		
		while [ "x$MANAGER_BASE_URL" = "x" ]
		do
			read -p "Nagios Log Server Manager URL (e.g., http://mylogserver.domain.com/nagioslogserver): " MANAGER_BASE_URL
			MANAGER_BASE_URL="${MANAGER_BASE_URL/[[:space:]]//}"
		done
		while [ "x$NODE_NAME" = "x" ]
		do
			read -p "Choose a name for this node: " NODE_NAME
			NODE_NAME="${NODE_NAME/[[:space:]]//}"
		done
        while [ "x$API_TOKEN" = "x" ]
        do
                read -p "Enter an Administrative API token for $MANAGER_BASE_URL: " API_TOKEN
                API_TOKEN="${API_TOKEN/[[:space:]]//}"
        done
	fi

}

test_cluster_connection() {
	
	if [ "$ADDING_NODE" = "true" ]; then
		# In an ideal world, we validate that we successfully added ourselves to the cluster.
		# However, some users will install this product on servers that don't immediately have network connectivity to other members of the cluster
		# i.e. they set up internal networking _after_ the install. 
		:
	fi
}

# Set the PHP timezone on RHEL-based systems
php_timezone() {
	local ZONE

	# Grab timezone from OS configuration (sets $ZONE)
	if [ "$dist" == "el7" ] || [ "$dist" == "el8" ] || [ "$dist" == "el9" ]; then
		ZONE=$(timedatectl status | sed -n '4p' | sed 's/.*: //' | sed 's/(.*)//')
	else
		if [ -r /etc/sysconfig/clock ]; then
			. /etc/sysconfig/clock
		fi
	fi

	# Set timezone if possible
	sed -i "s:^;date\.timezone =$:date.timezone = $ZONE:" "$phpini" || true
}

# Install SourceGuardian PHP extension
install_sourceguardian() {
	local phpver ixedfile entry zipfile

	# Get PHP version
	phpver=$(php -v | head -n 1 | cut -d ' ' -f 2 | cut -d . -f 1,2)

	ixedfile="ixed.$phpver.lin"
	entry="extension=$ixedfile"

	if [ "$arch" = "x86_64" ]; then
		zipfile="sourceguardian/loaders.linux-x86_64.zip"
	else
		echo "Error: Cannot install on 32bit systems"
		exit 1
	fi


	# Make sure we are using the exact php extension dir
	if [ `command -v php-config` ]; then
		php_extension_dir=$(php-config --extension-dir)
	fi

	# Extract SourceGuardian extension to the proper directory
	unzip -o "$zipfile" "$ixedfile" -d "$php_extension_dir"

	if [ -f "$php_extension_dir/$ixedfile" ]; then
		echo "Sourceguardian extension found for PHP version $phpver"
	else
		error "No valid Sourceguardian extension found for PHP" \
			"version $phpver"
	fi

	if grep -q "$entry" "$phpini" "$phpconfd"/*; then
		echo "Sourceguardian extension already in php.ini"
	else
		echo "Adding Sourceguardian extension to php.ini"
		echo "$entry" > "$phpconfd/sourceguardian.ini"
	fi

	# Add to the CLI ini file if it exists (debian systems)
	if [ $phpcliini ]; then
		if grep -q "$entry" "$phpcliini"; then
			echo "Sourceguardian extension already in php.ini"
		else
			echo "Adding entry to PHP CLI php.ini file"
			echo "$entry" > "$phpcliini"
		fi
	fi
}

# Send local syslog to standard port
setup_local_syslog() {
	mkdir -p /var/lib/rsyslog

	get_ip

	dynamic_ip=$((ip addr | grep dynamic >/dev/null 2>/dev/null); echo $?);

	cp -r nagioslogserver/logserver.syslog /etc/rsyslog.d/nagioslogserver.conf
	if [ $dynamic_ip -eq 1 ]; then
		sed -i "s/Target=\"localhost\"/Target=\"$ip\"/g" /etc/rsyslog.d/nagioslogserver.conf
	fi
	if [ `command -v firewalld` ]; then
		systemctl restart rsyslog
	else 
		service rsyslog restart
	fi
}


# Open the specified TCP ports
open_tcp_ports() {
	local chain rulenum

	if [ "$dist" == "el7" ] || [ "$dist" == "el8" ] || [ "$dist" == "el9" ]; then
		if [ `command -v firewalld` ]; then
			set +e
			for port; do
				firewall-cmd --zone=public --add-port="${port/:/-}"/tcp --permanent
			done
			firewall-cmd --reload
			set -e
		fi
	else
		# determine information for the rules
		chain=$(iptables -L | awk '/^Chain.*INPUT/ {print $2; exit(0)}')
		rulenum=$((`iptables -L $chain | wc -l` - 2))

		# test to make sure we aren't using less than the minimum 1
		if [ $rulenum -lt 1 ]; then rulenum=1; fi

		# add the rules
		for port; do
			iptables -I "$chain" "$rulenum" -m state --state NEW -m tcp \
				-p tcp --dport "$port" -j ACCEPT
		done

		# save changes
		service iptables save
		service iptables restart
	fi
}

# Open the specified UDP ports
open_udp_ports() {
	local chain rulenum

	if [ "$dist" == "el7" ] || [ "$dist" == "el8" ] || [ "$dist" == "el9" ]; then
		if [ `command -v firewalld` ]; then
			set +e
			for port; do
				firewall-cmd --zone=public --add-port="${port/:/-}"/udp --permanent
			done
			firewall-cmd --reload
			set -e
		fi
	else
		# determine information for the rules
		chain=$(iptables -L | awk '/^Chain.*INPUT/ { print $2; exit(0)}')
		rulenum=$((`iptables -L $chain | wc -l` - 2))

		# test to make sure we aren't using less tha n the minimum 1
		if [ $rulenum -lt 1 ]; then rulenum=1; fi

		# add the rules
		for port; do
			iptables -I "$chain" "$rulenum" -m state --state NEW -m udp \
					 -p udp --dport "$port" -j ACCEPT
		done
			
		# save changes
		service iptables save
		service iptables restart
	fi
}

# Disable SELinux on RHEL-based systems
disable_selinux() {
	if selinuxenabled; then
		setenforce 0
		cat >/etc/selinux/config <<-EOF
			# This file controls the state of SELinux on the system.
			# SELINUX= can take one of these three values:
			#     enforcing - SELinux security policy is enforced.
			#     permissive - SELinux prints warnings instead of enforcing.
			#     disabled - No SELinux policy is loaded.
			SELINUX=disabled
			# SELINUXTYPE= can take one of these two values:
			#     targeted - Targeted processes are protected,
			#     mls - Multi Level Security protection.
			SELINUXTYPE=targeted
			# SETLOCALDEFS= Check local definition changes
			SETLOCALDEFS=0
		EOF
	fi
}

# Enable services at boot and make sure it's running
rc() {
	for svc; do
		service "$svc" restart > /dev/null 2>&1 &
		chkconfig --level    35 "$svc" on
		chkconfig --level 01246 "$svc" off
	done
}

# If the system is Red Hat, make sure it's registered & has the appropriate
# channels enabled.
valid_rhel() {
	if [ -x /usr/sbin/rhn_check ] && ! /usr/sbin/rhn_check 2>/dev/null; then
		if [ -x /usr/bin/subscription-manager ] && [[ ! -z $(subscription-manager list | grep Status: | grep -qF Subscribed) ]]; then
			error "your Red Hat installation is not registered or does" \
				  "not have proper entitlements. Please register or" \
				  "enable entitlements at rhn.redhat.com."
		fi
	fi
}

# If the system is Red Hat, make sure it's subscribed to the specified channel.
# The first argument is a regexp to match the channel's label (e.g.
# rhel-.*-server-optional-6). The second argument is a human readable name for
# the channel (e.g. Optional).
has_rhel_channel() {
	if [ "$distro" = "RedHatEnterpriseServer" ] && ! rhn-channel -l | grep -q "$1"; then
		error "please add the '$2' channel to your Red Hat systems" \
			"subscriptions. You can do so in the Red Hat Network" \
			"web interface or by using the rhn-channel command."
	fi
}

# Run some standard checks - root user check, $PATH variable, RHEL validation
std_checks() {
	path_is_ok() {
		echo "$PATH" \
		| awk 'BEGIN{RS=":"} {p[$0]++} END{if (p["/sbin"] && p["/usr/sbin"]) exit(0); exit(1)}'
	}

	if [ $(id -u) -ne 0 ]; then
		error "This script needs to be run as root/superuser."
	fi

	if [ "$distro" == "RedHatEnterpriseServer" ]; then
		valid_rhel
	fi

	if ! path_is_ok; then
		echo "Your system \$PATH does not include /sbin and" \
			"/usr/sbin. This is usually the result of installing" \
			"GNOME rather than creating a clean system."
		echo "Adding /sbin and /usr/sbin to \$PATH."
		PATH="$PATH:/sbin:/usr/sbin"
	fi
	unset -f path_is_ok
}

# Detect OS & set global variables for other commands to use.
# OS variables have a detailed long variable, and a "more useful" short one:
# distro/dist, version/ver, architecture/arch. If in doubt, use the short one.
set_os_info() {
	if [ `uname -s` != "Linux" ]; then
		error "Unsupported OS detected. Currently only Linux distributions can be detected."
	fi

	# Get OS & version
	if which lsb_release &>/dev/null; then
		distro=`lsb_release -si`
		version=`lsb_release -sr`
	elif [ -r /etc/redhat-release ]; then

		if rpm -q centos-linux-release || rpm -q centos-stream-release || rpm -q centos-release; then
			distro=CentOS
		elif rpm -q sl-release; then
			distro=Scientific
		elif [ -r /etc/oracle-release ]; then
			distro=OracleServer
		elif rpm -q cloudlinux-release; then
			distro=CloudLinux
		elif rpm -q fedora-release; then
			distro=Fedora
		elif rpm -q redhat-release || rpm -q redhat-release-server || rpm -q sles_es-release-server || rpm -q sles_es-release; then
			distro=RedHatEnterpriseServer
		fi >/dev/null

		version=`grep -ve ^# /etc/redhat-release|sed 's/.*release \([0-9.]\+\).*/\1/'`
	else
		# Release is not RedHat or CentOS, let's start by checking for SuSE
		# or we can just make the last-ditch effort to find out the OS by sourcing os-release if it exists
		if [ -r /etc/os-release ]; then
			source /etc/os-release
			if [ -n "$NAME" ]; then
				distro=$NAME
				version=$VERSION_ID
			fi
		fi
	fi

	# Everyone likes to change their NAME now
	if [[ $distro == *"Debian"* ]]; then
		distro="Debian"
	elif [[ $distro == *"CentOS"* ]]; then
		distro="CentOS"
	elif [[ $distro == *"RedHatEnterprise"* ]]; then
		distro="RedHatEnterpriseServer"
	elif [[ $distro == *"redhatenterprise"* ]]; then
		distro="RedHatEnterpriseServer"
	fi

	# Add patch level to the version of SLES (because they don't...)
	if [ "$distro" == "SUSE LINUX" ]; then
		if [ -r /etc/SuSE-release ]; then
			patchlevel=$(cat /etc/SuSE-release | cut -d ' ' -f 3 -s | sed -n 3p)
			version="$version.$patchlevel"
		fi
	fi

	# Verify that we have a distro now
	if [ -z "$distro" ]; then
		echo "ERROR: Could not determine OS. Please make sure lsb_release is installed or your OS info is in /etc/os-release." >&2
		exit 1
	fi

	ver="${version%%.*}"

	case "$distro" in
		CentOS | RedHatEnterpriseServer | OracleServer | CloudLinux )
			dist="el$ver"
			;;
		Debian )
			dist="debian$ver"
			;;
		* )
			dist=$(echo "$distro$ver" | tr A-Z a-z)
	esac

	architecture=`uname -m`

	# i386 is a more useful value than i686 for el5, because repo paths and
	# package names use i386
	if [ "$dist $architecture" = "el5 i686" ]; then
		arch="i386"
	else
		arch="$architecture"
	fi

	httpd='httpd'
	mysqld='mysqld'

	apacheuser='apache'
	apachegroup='apache'
	nagiosuser='nagios'
	nagiosgroup='nagios'
	nagioscmdgroup='nagcmd'
	ntpd='ntpd'
	crond="crond"

	phpini='/etc/php.ini'
	phpconfd='/etc/php.d'
	httpdconfdir='/etc/httpd/conf.d'
	mrtgcfg='/etc/mrtg/mrtg.cfg'
	mibsdir='/usr/share/snmp/mibs'

	# This is set later by php-config if possible
	php_extension_dir='/usr/lib64/php/modules'

	case "$dist" in
		el6 | el7 | el8 | el9 )
			if [ "$dist" == "el7" ]; then
				mysqld="mariadb"
			fi
			if [ "$dist" == "el8" ] || [ "$dist" == "el9" ]; then
				ntpd="chronyd"
			fi
			;;
		ubuntu22 | ubuntu24 | debian10 | debian11 | debian12 )
			if [ "$dist" == "debian11" ]; then
				phpini="/etc/php/7.4/apache2/php.ini"
				phpconfd="/etc/php/7.4/apache2/conf.d"
				phpconfdcli="/etc/php/7.4/cli/conf.d"
				phpcliini="/etc/php/7.4/cli/php.ini"
			elif [ "$dist" == "ubuntu22" ]; then
				phpini="/etc/php/8.1/apache2/php.ini"
				phpconfd="/etc/php/8.1/apache2/conf.d"
				phpconfdcli="/etc/php/8.1/cli/conf.d"
				phpcliini="/etc/php/8.1/cli/php.ini"
			elif [ "$dist" == "debian12" ]; then
				phpini="/etc/php/8.2/apache2/php.ini"
				phpconfd="/etc/php/8.2/apache2/conf.d"
				phpconfdcli="/etc/php/8.2/cli/conf.d"
				phpcliini="/etc/php/8.2/cli/php.ini"
			elif [ "$dist" == "ubuntu24" ]; then
				phpini="/etc/php/8.3/apache2/php.ini"
				phpconfd="/etc/php/8.3/apache2/conf.d"
				phpconfdcli="/etc/php/8.3/cli/conf.d"
				phpcliini="/etc/php/8.3/cli/php.ini"
			else
				phpini="/etc/php5/apache2/php.ini"
				phpconfd="/etc/php5/apache2/conf.d"
				phpconfdcli="/etc/php5/cli/conf.d"
				phpcliini="/etc/php5/cli/php.ini"
			fi
			mibsdir="/usr/share/mibs"
			httpdconfdir="/etc/apache2/sites-available"
			apacheuser="www-data"
			apachegroup="www-data"
			httpdconf="/etc/apache2/apache2.conf"
			httpdroot="/var/www/html"
			httpd="apache2"
			ntpd="ntp"
			crond="cron"
			mysqld="mysql"

			if [ "$dist" == "debian10" ] || [ "$dist" == "debian11" ] || [ "$dist" == "debian12" ]; then
				mysqld="mariadb"
			fi
			if [ "$dist" == "debian12" ] ; then
				ntpd="ntpsec"
			fi
			;;
	esac
}

# Get curent php version 5.3 or higher
# - The $phpversion value will equal 1 if PHP >= 5.3.0
get_php53_or_higher() {
	phpversion=$(php -r "print version_compare(PHP_VERSION, '5.3.0');")
}

# Update sudoers file if it hasn't already been updated
sudoers() {
	sudofile="nagioslogserver/nagioslogserver.sudoers"

	# Remove old sudoers entries
	grep -v NAGIOSLOGSERVER /etc/sudoers > /etc/sudoers.new
	mv -f /etc/sudoers.new /etc/sudoers

	# Remove TTY requirement
	sed -i 's/Defaults    requiretty/#Defaults    requiretty/g' /etc/sudoers

	# Apply www-data instead of apache and use the special sudoers file
	if [ "$distro" == "Ubuntu" ] || [ "$distro" == "Debian" ]; then
		cp $sudofile nagioslogserver/nagioslogserver.deb.sudoers
		sed -i 's/apache/www-data/g' nagioslogserver/nagioslogserver.deb.sudoers
		sudofile="nagioslogserver/nagioslogserver.deb.sudoers"
	fi

	# Add new sudoers entries and set permissions
	cat $sudofile >> /etc/sudoers
	chmod 440 /etc/sudoers
}

# Currently supporting:
#	CentOS / RHEL / Oracle / 8, 9 (Except CentoS Stream 8)
#
# Will install on:
# 	Ubuntu 22, 24(LTS ONLY)
#	Debian 11, 12
#
# No longer supported:
#	CentOS / RHEL 5, 6, 7, CentOS Stream 8
#   Ubuntu 14, 16, 18, 20
#   Debian 8, 9, 10
# Checks if the system is able to be installed on
do_install_check() {

	# Check if NLS is already installed
	if [ -f "/usr/local/nagioslogserver/.installed" ]; then
		echo "It looks like Nagios Log Server is already installed. Use ./upgrade to upgrade instead." >&2
		exit 1
	fi

	# Check if any other product is already installed
	products=( nagiosna nagiosxi nagiosfusion )
	for dir in "{$products[@]}"; do
		if [ -d "/usr/local/$dir" ]; then
			echo "It looks like another product ($dir) is already installed on this system." >&2
			exit 1
		fi
	done

	if [ "$dist" == "el8" ] || [ "$dist" == "el9" ]; then
		echo "Installing on $distro $version"
	elif [ "$dist" == "ubuntu22" ] || [ "$dist" == "ubuntu24" ]; then
		echo "Installing on $distro $version"
	elif [ "$dist" == "debian11" ] || [ "$dist" == "debian12" ]; then
		echo "Installing on $distro $version"
	else
		echo "" >&2
		echo "$dist is not currently supported. Please use one of the following distros:" >&2
		echo "  RHEL or Oracle 8 or 9, or CentOS 9" >&2
		echo "  Ubuntu 22, or 24 LTS" >&2
		echo "  Debian 11 or 12" >&2
		echo "" >&2
		exit 1
	fi
}

# Checks to verify that the upgrade can be performed
do_upgrade_check() {

	# Check if NLS is already installed
	if [ ! -d $proddir ]; then
		echo "Could not find an installed version of Nagios Log Server."
		exit 0
	fi

	# Check valid versions
	if [ "$dist" == "el8" ] || [ "$dist" == "el9" ]; then
		echo "Upgrading on $distro $version"
	elif [ "$dist" == "ubuntu22" ]|| [ "$dist" == "ubuntu24" ]; then
		echo "Upgrading on $distro $version"
	elif [ "$dist" == "debian11" ] || [ "$dist" == "debian12" ]; then
		echo "Upgrading on $distro $version"
	else
		echo "" >&2
		echo "$dist is not currently supported. Please use one of the following distros:" >&2
		echo "  RHEL or Oracle 8 or 9, or CentOS 9" >&2
		echo "  Ubuntu 22 or 24 LTS" >&2
		echo "  Debian 11 or 12" >&2
		echo "" >&2
		exit 1
	fi

}

# sets OpenSearch cluster to reallocate data across shards
elasticsearch_allow_reallocate() {	

	# only actually do the sed if the correct config file exists
	if test -f /usr/local/nagioslogserver/elasticsearch/config/elasticsearch.yml; then
		if ! grep -q "cluster.routing.allocation.enable : all" /usr/local/nagioslogserver/elasticsearch/config/elasticsearch.yml; then
			echo "Setting OpenSearch to allow shard reallocation"
			sed -i '/^# cluster.routing.allocation.node_concurrent_recoveries:.*/a cluster.routing.allocation.enable: all\nindex.unassigned.node_left.delayed_timeout: 0' /usr/local/nagioslogserver/elasticsearch/config/elasticsearch.yml			
		fi
	fi
}

# Initialize installation - run basic checks and detect OS info
set_os_info
std_checks


update_java_logging_configs() {
	cp /usr/local/nagioslogserver/logstash/config/jvm.options /usr/local/nagioslogserver/logstash/config/jvm.options.bak
	cp subcomponents/logstash/mods/jvm.options /usr/local/nagioslogserver/logstash/config/jvm.options
	cp /usr/local/nagioslogserver/logstash/config/log4j2.properties /usr/local/nagioslogserver/logstash/config/log4j2.properties.bak
	cp subcomponents/logstash/mods/log4j2.properties /usr/local/nagioslogserver/logstash/config/log4j2.properties
	cp /usr/local/nagioslogserver/opensearch/config/jvm.options /usr/local/nagioslogserver/opensearch/config/jvm.options.bak
	cp subcomponents/opensearch/mods/opensearch/jvm.options /usr/local/nagioslogserver/opensearch/config/jvm.options
	cp /usr/local/nagioslogserver/opensearch/config/log4j2.properties /usr/local/nagioslogserver/opensearch/config/log4j2.properties.bak
	cp subcomponents/opensearch/mods/opensearch/log4j2.properties /usr/local/nagioslogserver/opensearch/config/log4j2.properties
	sed -i s/rootLogger.level\ =\ info/rootLogger.level\ =\ warn/1 /usr/local/nagioslogserver/opensearch/config/log4j2.properties
	sed -i s/^#\ log.level:\ info$/log.level:\ warn/1 /usr/local/nagioslogserver/logstash/config/logstash.yml
}