<?php  if ( ! defined('BASEPATH')) exit('No direct script access allowed');

class Dashboard extends LS_Controller
{
    private $offset = 0;

    function __construct()
    {
        parent::__construct();

        // Make sure that user is authenticated no matter what page they are on
        require_install();
        if (!$this->users->logged_in()) { $this->users->redirect_to_login(); }

        // Fix key value timezone offset to match server time
        $x = date_default_timezone_get();
        if ($x) {
            $dt = new DateTime("now", new DateTimeZone($x));
            $this->offset = $dt->getOffset()*1000;
        }
    }

    // Logs out user
    function logout()
    {
        // Log the user out
        $logout = $this->users->logout();

        // Redirect them to the login page
        $this->session->set_flashdata('message', $this->users->message);
        redirect('login');
    }

    // Basic dashboard
    public function index()
    {
        // reroute to simple homepage if user is a simple user
        if ($this->users->get_user()['simple_mode'] == "true") {
            $this->init_page(_("Home"), 'simple_home');
            $this->load->view('simple_home', $this->data);
        }
        else {
            $this->data['dashboardId'] = 'home';
            $this->init_page(_("Home"), 'dashboard'); 
            $this->load->view('dashboard', $this->data);
        }
    }

    public function modernjs_home() {
        $this->init_page(_("ModernJS Home"), 'modernjs_home');
        $this->load->view('modernjs_home', $this->data);
    }

    public function dashboard(): void {
        $path = str_replace("/dashboard", "", $_SERVER['PATH_INFO']);
        $path = str_replace("%20", "+", $path);
        $path = str_replace(" ", "+", $path);
        $path_array = explode('/',$path);

        $this->data['dashboardId'] = $path_array[1];
        $this->init_page(_("ModernJS Home"), 'modernjs_home');
        $this->load->view('dashboard', $this->data);
    }

    public function show_dash()
    {
        $this->init_page(_("Dashboard"), 'dashboard', array());

        $this->data['user'] = $this->users->get_user();

        $this->load->view('dashboard', $this->data);
    }
    
    public function dashlet()
    {
        $this->data = array();
        $this->load->view('dashlet/dashlet', $this->data);
    }
    
    public function do_update_check()
    {
        $this->init_page();

        // Do an update check
        if (is_admin()) {
            $xml = do_update_check();
        }
        $update_check = simplexml_load_string(base64_decode($xml));

        $text = '';
        $icon = '';
        if (intval($update_check->update_available)) {
            $text =  _("New version available!");
            $icon = '<div class="material-symbols-outlined tw-text-warning large">priority_high</div>';
        } else {
            $text = _("You're running the latest version of Nagios Log Server");
            $icon = '<div class="material-symbols-outlined tw-text-success large">check</div>';
        }
        if (intval($update_check->update_available)) {
            $text .= _(" Version ") . strval($update_check->update_version->version) . _(" is the latest version available.");
        }

        $html = '<tooltip-provider>
                    <tooltip data-position="bottom" data-delay="100">
                        <tooltip-trigger>
                            '.($icon).'
                        </tooltip-trigger>
                        <tooltip-content>
                            <div>'._($text).'</div>
                        </tooltip-content>
                    </tooltip>
                </tooltip-provider>';

        print $html;
    }

    // Basic user profile (can't do much here - requires auth to change certain things)
    // - for api key mostly
    public function profile()
    {
        $this->init_page(_("My Profile"), 'profile');
        $this->data['csrf'] = array(
            'name' => $this->security->get_csrf_token_name(),
            'hash' => $this->security->get_csrf_hash()
        );

        $this->data['user'] = $this->users->get_user();
        $this->data['languages'] = get_languages();
        $this->data['themes'] = get_themes();
        $this->data['spacings'] = get_spacings();

        $this->load->view('profile', $this->data);
    }

    public function newkey()
    {
        $redirect_to_help = $this->input->post('help', FALSE);
        
        if (!$this->data['demo_mode']) {
            // Generate a new API key
            $newkey = sha1(uniqid() . 'nagioslogserver');

            // Update new key in database
            $user_id = $this->session->userdata('user_id');
            $verify_user_id = $this->input->post('user_id_verify', TRUE);

            if ($user_id == $verify_user_id) {
                // Update the new api key
                $data = array("apikey" => $newkey);
                $this->users->update_user($user_id, $data);
            }

            // Set session data
            $this->session->set_userdata('apikey', $newkey);
        }

        if ($redirect_to_help) {
            redirect('help/api');
        }
        redirect('profile');
    }

    public function setlanguage()
    {
        $user_id = $this->session->userdata('user_id');
        $language = $this->input->post('language');

        // Update db
        $data = array("language" => $language);
		$this->users->update_user($user_id, $data);

        // Update session
        $this->session->set_userdata('language', $language);
    }

    public function set_theme()
    {
        $user_id = $this->session->userdata('user_id');
        $theme = $this->input->post('theme');

        // Update db
        $data = array("theme" => $theme);
        $this->users->update_user($user_id, $data);

        // Update session
        $this->session->set_userdata('theme', $theme);
    }

    public function set_spacing()
    {
        $user_id = $this->session->userdata('user_id');
        $spacing = $this->input->post('spacing');

        // Update db
        $data = array("spacing" => $spacing);
        $this->users->update_user($user_id, $data);

        // Update session
        $this->session->set_userdata('spacing', $spacing);
    }

    public function set_mode()
    {
        $user_id = $this->session->userdata('user_id');
        $simple_mode = $this->input->post('simple_mode');

        // Update db
        $data = array("simple_mode" => $simple_mode);
        $this->users->update_user($user_id, $data);

        // Update session
        $this->session->set_userdata('simple_mode', $simple_mode);
    }

    public function set_no_simple_help()
    {
        $user_id = $this->session->userdata('user_id');
        $no_simple_help = ($this->input->post('no_simple_help') == "true") ? true : false;

        // Update db
        $data = array("no_simple_help" => $no_simple_help);
        $this->users->update_user($user_id, $data);
    }

    // TODO: This will need some work
    public function export()
    {
        $query = $this->input->get_post('q');
        $limit = $this->input->get_post('limit');
        $fields = $this->input->get_post('fields');
        $fields = explode(',', $fields);

        $csv_use_local_timezone = get_option("csv_use_local_timezone");
        $timezone = new DateTime('now');
        $timezone = $timezone->getTimezone();

        if (empty($query) || empty($fields)) {
            print _("Cannot make CSV with the query and fields given.");
            print "<p>q = $query</p>";
            print "<p>fields = ".print_r($fields, true)."</p>";
            die();
        }

        $this->load->helper('multitenancy');
        $query = str_replace(array(" and ", " or "),array(" AND ", " OR "), $query);
        $query = multitenancy_limit_query($query);

        // Do the query
        $backend = new Elasticsearch(array('index' => '/'));
        $data = $backend->backend_call("_search", 'GET', $query);

        // Output header for csv
        header("Content-type: application/octet-stream");
        header("Content-Disposition: attachment; filename=\"export.csv\"");

        // Output CSV format
        print implode(',', $fields) . "\n";
        foreach ($data['hits']['hits'] as $hit) {
            $tmp = array();
            foreach ($fields as $i) {

                $field = "";
                // Check for a nested field - usually geoip related.
                if (strpos($i, '.') === false) {
                    
                    if (@isset($hit['_source'][$i])) {
                        $field = $hit['_source'][$i];
                    } else {
                        $field = $hit[$i];
                    }
                }
                else {
                    // Field is nested - recursively navigate to the actual value
                    $keys = explode('.', $i);
                    $field = $hit['_source'];
                    for ($j = 0; $j < count($keys); $j++) {
                        if (@isset($field[$keys[$j]])) {
                            $field = $field[$keys[$j]];
                        }
                    }
                }

                // This converts the timestamp from UTC to the system timezone (if the global option is set)
                if ($csv_use_local_timezone && $i === "@timestamp") {
                    $as_datetime = new DateTime($field);
                    $as_datetime->setTimezone($timezone);
                    $field = $as_datetime->format('c');
                }

                $tmp[] = '"'.trim(str_replace(array("\r", "\n", "&apos;", "\""), array(" ", " ", "'", "\"\""), html_entity_decode($field))).'"';
            }
            print implode(',', $tmp) . "\n";
        }
    }

    public function get_natural_language_query_status() {
        $natural_language_query = get_option("natural_language_query");
        echo json_encode(['value' => $natural_language_query]);
    }

    public function natural_language_to_query() {
        $input = isset($_GET['input']) ? $_GET['input'] : '';
        $input = escapeshellarg($input);
        $current_fields = isset($_GET['current_fields']) ? $_GET['current_fields'] : '';
        $current_fields = escapeshellarg($current_fields);
    
        $api_key = escapeshellarg(trim(get_option("ai_api_key")));
        $model = escapeshellarg(trim(get_option("ai_provider")));
        $self_host_ip_address = escapeshellarg(trim(get_option("self_host_ip_address")));
        $self_host_port = escapeshellarg(trim(get_option("ai_port")));
        
        $script =  '/usr/local/nagioslogserver/scripts/generate_log_query.py';
    
        if ($model == "self_hosted") {
            $command = "python3.9 $script --model \"$model\" --provider_address \"$self_host_ip_address\" --provider_port \"$self_host_port\" --natural_language_query \"$input\" --current_fields \"$current_fields\"";
        } else {
            $command = "python3.9 $script --api_key \"$api_key\" --model \"$model\" --natural_language_query \"$input\" --current_fields \"$current_fields\"";
        }
    
        $output = shell_exec($command);
        
        $lucene_query = trim($output);

        $es = new Elasticsearch(array('index' => '/'));
        $query = '{
            "query": {
                "query_string": {
                    "query": "' . $lucene_query . '"
                }
            }
        }';

        $result = $es->backend_call('_validate/query', 'GET', $query);
        $is_valid = isset($result['valid']) ? $result['valid'] : false;
        // holy shit this works https://www.elastic.co/guide/en/elasticsearch/reference/current/search-validate.html
        header('Content-Type: application/json');
        echo json_encode([
            'query' => $lucene_query,
            'is_valid' => $is_valid
        ]);
    }
}
