<?php

// =====================================
//  LDAP / AD Connection(s)
// =====================================

function open_server_conn($server_id)
{
    $ci =& get_instance();

    // Start by determining what to connect to
    $raw_servers = $ci->config_option->get('auth_servers', true);
    if (!empty($raw_servers)) {
        $servers = unserialize(base64_decode($raw_servers));
    } else {
        $servers = array();
    }

    foreach ($servers as $s) {
        if ($s['id'] == $server_id) {
            $server = $s;
            break;
        }
    }

    if (empty($server)) {
        return false;
    }

    // Connect to the server...
    if ($server['type'] == "ldap") {

        $ci->load->library('BasicLDAP');

        $use_ssl = false;
        $use_tls = false;

        if ($server['encryption'] == "ssl") {
            $use_ssl = true;
        } else if ($server['encryption'] == "tls") {
            $use_tls = true;
        }

        $ldap = new BasicLDAP($server['host'], $server['port'], $server['basedn'], $server['encryption']);
        return $ldap;

    } else if ($server['type'] == "ad") {

        $ci->load->library('adLDAP/AdLDAP');

        $use_ssl = false;
        $use_tls = false;

        if ($server['encryption'] == "ssl") {
            $use_ssl = true;
        } else if ($server['encryption'] == "tls") {
            $use_tls = true;
        }

        $controllers = explode(',', $server['controllers']);

        // Create the adLDAP object...
        $options = array('account_suffix' => $server['suffix'],
                         'base_dn' => $server['basedn'],
                         'domain_controllers' => $controllers,
                         'use_ssl' => $use_ssl,
                         'use_tls' => $use_tls);

        try {
            $ad = new AdLDAP($options);
            return $ad;
        } catch (adLDAPException $e) {
            return false;
        }

    }

    return false;
}

function create_auth_connection()
{
    $ci =& get_instance();
    $username = $ci->session->userdata('auth_import_username');
    $password = $ci->session->userdata('auth_import_password');
    $server_id = $ci->session->userdata('auth_import_server_id');
    $conn = open_server_conn($server_id);

    // Try to check authentication when creating a new object
    try {
        $x = $conn->authenticate($username, $password);
        if (!$x) {
            ldap_get_option($conn->getLdapConnection(), LDAP_OPT_ERROR_STRING, $out);
            if (empty($out)) {
                $ad_error = _("Could not connect to the LDAP server selected.");
            } else {
                $ad_error = $out;
            }
            return false;
        }
        return $conn;
    } catch (Exception $ex) {
        $ad_error = $ex->getMessage();
        return false;
    }
}

// ====================================
//  API Helper Functions
// ====================================

function grab_ad_folders($folder="", $type="container", $search="", $target_page=1, $page_size=50)
{
    $conn = create_auth_connection();
    $output = false;
    $base_dn = $conn->getBaseDn();

    if ($type == "organizationalUnit") {
        if ($conn->type == "ad") {
            $list_array = $conn->folder()->listing($folder, AdLDAP::ADLDAP_FOLDER, false, null, $search, $target_page, $page_size);
        } else {
            $list_array = $conn->folder_listing($folder, BasicLDAP::LDAP_FOLDER, $search, $target_page, $page_size);
        }
        $output = array(
            'entries' => check_validity($list_array['entries']),
            'target_page' => $target_page,
            'page_count' => $list_array['pageCount'],
            'page_size' => $list_array['pageSize']
        );
    } else if ($type == "container" || $type == "nsContainer") {
        if ($conn->type == "ad") {
            $list_array = $conn->folder()->listing($folder, adLDAP::ADLDAP_FOLDER, false, null, $search, $target_page, $page_size);
        } else {
            $list_array = $conn->folder_listing($folder, basicLDAP::LDAP_FOLDER, $search, $target_page, $page_size);
        }
        $output = array(
            'entries' => check_validity($list_array['entries']),
            'target_page' => $target_page,
            'page_count' => $list_array['pageCount'],
            'page_size' => $list_array['pageSize']
        );
    } else if ($type == "group") {
        $folder = grab_array_var($folder, "0");
        $ad_array = $conn->group()->members($folder, null, $search, $target_page, $page_size);
        $output = array(
            'entries' => $ad_array,
            'target_page' => $target_page,
            'page_count' => 1,
            'page_size' => 50
        );
    }

    if ($output) {
        $output['base_dn'] = $base_dn;
    }
    return $output;
}

// ====================================
//  HTML Generating Functions
// ====================================

function display_nav_window($array_to_enum, $new_list=0)
{
    // Hide some folders that shouldn't be shown because they are very VERY unlikely to have users in them
    // unless someone likes putting their users in strange places...
    $dont_show = array("System", "Program Data", "ForeignSecurityPrincipals", "Managed Service Accounts");
    $output_html = '';
    if ($new_list) {
        $output_html .= '<ul class="ad-list folder-list">';
    } else {
        $output_html .= '<ul class="ad-list sub-list">';
    }
    $base_ou = grab_base_ou($array_to_enum['entries']);

    if (!($array_to_enum['entries'] == false)) {
        foreach ($array_to_enum['entries'] as $obj) {
            if (is_array($obj)) {
                $path = grab_path($obj);
                if ($base_ou) {
                    $path = json_encode(array_diff($path, $base_ou));
                }
                else {
                    $path = json_encode($path);
                }
                $dn = grab_dn($obj);
                $type = grab_type($obj);

                if ($type == "organizationalUnit" || $type == "container" || $type == "group" || $type == "nsContainer") {
                    
                    // Skip if the object is something we don't need to display
                    if (in_array($dn, $dont_show)) {
                        continue;
                    }

                    if ($type == "group") { $image = "group.png"; }
                    if ($type == "container" || $type == "nsContainer") { $image = "folder.png"; }
                    if ($type == "organizationalUnit") { $image = "folder_page.png"; }
                    $output_html .=
                    '<li class="tw-py-2">
                        <span class="ad-folder tw-flex tw-gap-2 tw-cursor-pointer" data-path=\'' . $path . '\' data-type="' . $type . '">
                            <img class="table-icon" src="' . base_url('media/icons/' . $image) . '">
                            ' . $dn . '
                        </span>
                    </li>';
                }
            }
        }
    }
    $output_html .= '</ul>';
    echo json_encode(Array($output_html, $base_ou));
}

function display_users($array_to_enum, $location, $type, $selected, $search = "")
{
    // List of usernames not to show...
    $dont_show = array("krbtgt");
    if ($location) {
       $location = grab_array_var($location, "0");
    }
    $conn = create_auth_connection();
    $base_dn = $conn->getBaseDn();
    $person_exists = false;
    $all_checked = true;
    $printed = false;

    $current_page = $array_to_enum['target_page'];
    $page_count = $array_to_enum['page_count'];
    $page_size = $array_to_enum['page_size'];

    $user_png = base_url("media/icons/user.png");
    $add_user_text = _('Add user');
    $show_full_dn_text = _("Show full DN (distinguished name)");

    if (!empty($array_to_enum['entries'])) {
        if ($type == "group") {
            foreach ($array_to_enum['entries'] as $username)
            {
                $person_exists = true;
                $userinfo = $conn->user()->info($username, array("displayname"));
                $displayname = $userinfo[0]["displayname"][0];
                $dn = $userinfo[0]["dn"];
                $full_display_name = ($username == $displayname) ? $username : $displayname . ' (' . $username . ')';
                $checkbox_value = ($conn->type == "ad")  ? $username : $dn;
                $is_checkbox_checked = '';
                if (in_array($dn, $selected)) {
                     $is_checkbox_checked = "checked";
                } else {
                    $all_checked = false;
                }

                $users_html .= '
                <li class="tw-px-8">
                    <div class="tw-flex tw-items-center tw-space-x-2">
                        <input type="checkbox" class="ad-checkbox ui-checkbox-group-item" value="'.$checkbox_value.'"'.$is_checkbox_checked.'>
                        <label class="checkbox tw-flex" for="ad-checkbox">
                            <img class="table-icon" src="'.$user_png.'" border="0" alt="'.$add_user_text.'" title="'.$add_user_text.'" style="">
                            '.$full_display_name.'
                        </label>
                    </div>
                    <div class="tw-flex tw-items-center tw-gap-2 tw-pl-1">
                        <i class="fa fa-plus-square-o user-toggle-show-dn" title="'.$show_full_dn_text.'"></i>
                        <div class="user-dn hide">DN: '.$dn.'</div>
                    </div>
                </li>
                ';
            }
        } else {
            foreach ($array_to_enum['entries'] as $obj)
            {
                if (is_array($obj)) {

                    $valid_units = array('person', 'inetorgperson', 'organizationalperson', 'shadowaccount', 'posixaccount');
                    $type = grab_type($obj);
                    if (in_array(strtolower($type), $valid_units)) {
                        $username = grab_user_name($type, $obj);
                        $dn = grab_full_dn($obj);
                        $displayname = grab_dn($obj);

                        if (in_array($username, $dont_show)) {
                            continue;
                        }
                        $username = (isset($username)) ? $username : $displayname;
                        $full_display_name = ($username == grab_dn($obj)) ? $username : $displayname . ' (' . $username . ')';
                        $image = grab_image_for_type($obj);
                        $person_exists = true;
                        $checkbox_value = ($conn->type == "ad") ? $username : $dn;
                        $is_checkbox_disabled = (is_computer($obj)) ? 'disabled' : '';
                        $is_checkbox_checked = '';
                        if (in_array($dn, $selected)) {
                            $is_checkbox_checked = 'checked';
                        } else {
                             $all_checked = false;
                        }
                        $table_icon_image = base_url('media/icons/'.$image);

                        $users_html .= '
                        <li class="tw-px-8">
                            <div class="tw-flex tw-flex-row">
                                <div class="tw-flex tw-items-center tw-space-x-2">
                                    <input type="checkbox" class="ad-checkbox ui-checkbox-group-item" data-displayname="'.$displayname.'" value="'.$checkbox_value.'"'.$is_checkbox_disabled.' '.$is_checkbox_checked.'>                                        
                                    <label class="checkbox tw-flex">
                                        <img class="table-icon" src="'.$table_icon_image.'" border="0" alt="'.$add_user_text.'" title="'.$add_user_text.'">
                                        '.$full_display_name.'
                                    </label>
                                </div>
                                <div class="tw-flex tw-items-center tw-gap-2 tw-pl-1">
                                    <i class="fa fa-plus-square-o user-toggle-show-dn" title="'.$show_full_dn_text.'"></i>
                                </div>
                            </div>
                            <div class="user-dn hide">DN: '.$dn.'</div>
                        </li>
                        ';
                    }
                }
            }
        }
    }
    if ($person_exists) {
        $page_controls = get_page_controls($all_checked, $current_page, $page_count, $page_size, $search);
        $users_html = '<ul class="ad-list user-list">' . $page_controls . $users_html;
    } else {
        $page_controls = get_page_controls(false, $current_page, $page_count, $page_size, $search);
        if ($search != "") {
            $no_results_text = _("No users or computers found matching your search.");
            $search_tips = '<p class="tw-font-bold tw-p-2">'._("Search Tips:").'</p>
                <ul class="tw-p-2">
                    <li class="tw-list-disc tw-ml-4">'._("Try using the wildcard character (*)").'</li>
                    <li class="tw-list-disc tw-ml-4">'._("Search for a user's name, rather than their username").'</li>
                    <li class="tw-list-disc tw-ml-4">'._("Try different or fewer keywords").'</li>
                    <li class="tw-list-disc tw-ml-4">'._("Check your spelling").'</li>
                </ul>
            </li>';
        } else {
            $no_results_text = _("No users or computers found in this object.");
        }
        $users_html .= '
        <ul class="ad-list user-list">
            ' . $page_controls . '
            <li class="tw-p-2">'.$no_results_text.'</li>
            '.$search_tips.'
        </ul>
        ';
    }
    echo $users_html;
}

// ==============================
//  Random Helper Functions
// ==============================

function check_validity($ad_array) {
    $count = grab_array_var($ad_array, "count", 0);
    if (!($count == 0)) {
        return $ad_array;
    } else {
        return false;
    }
}

function grab_type($obj) {
    $item = grab_array_var($obj, "objectclass", "");
    if (!empty($item)) {
        $type = grab_array_var($item, "1", "");
        if (empty($type)) {
            $type = grab_array_var($item, "0", "");
        }
        return $type;
    }
}

function grab_user_name($type, $obj) {
    if ($type == "person") {
        $item = grab_array_var($obj, "samaccountname");
        if (!empty($item)) {
            return grab_array_var($item, "0", "");
        }
    } else if ($type == "inetOrgPerson") {
        $item = grab_array_var($obj, "uid");
        if (!empty($item)) {
            return grab_array_var($item, "0", "");
        }
    }
}

function grab_dn($obj) {
    $item = grab_array_var($obj, "dn");
    $item = str_replace(array('\,', '\2C'), '&#44;', $item);
    if (!($item == "")) {
        $dn = explode(",", $item);
        $value = explode("=", grab_array_var($dn, "0"));
        return grab_array_var($value, "1");
    }
}

function grab_full_dn($obj) {
    return grab_array_var($obj, "dn", grab_dn($obj));
}

function grab_path($obj) {
    $item = grab_array_var($obj, "dn", "");
    $item = str_replace(array('\,', '\2C'), '&#44;', $item);
    $path = array();
    
    if (!($item == "")) {
        $fully_qualified = explode(",", $item);
        foreach ($fully_qualified as $branch) {
            $value = explode("=", $branch);
            $id = grab_array_var($value, "0");
            if (strtoupper($id) == "OU" || strtoupper($id) == "CN") {
                $ou_location = grab_array_var($value, "1");
                if (!($ou_location == "")){
                    array_push($path, $ou_location);
                }
            }
        }
    }
    return $path;
}

function grab_image_for_type($obj)
{
    if (is_user($obj)) {
        $image = "user.png";
    }
    if (is_computer($obj)) {
        $image = "monitor.png";
    }
    return $image;
}

function is_computer($ad_array) {
    $comp = grab_array_var($ad_array["objectclass"], "4", "");
    if ($comp == "computer") {
        return true;
    }
    return false;
}

function is_user($ad_obj) {
    $ci =& get_instance();
    $type = $ci->session->userdata('auth_import_server_type');

    if ($type == "ad") {
        $comp = grab_array_var($ad_obj["objectclass"], "4", "");
        $person = grab_array_var($ad_obj["objectclass"], "1", "");
        if (empty($comp) && $person == "person") {
            return true;
        }
    } else if ($type == "ldap") {
        $valid_units = array('person', 'inetorgperson', 'organizationalperson', 'shadowaccount', 'posixaccount');
        $t = grab_type($ad_obj);
        if (in_array(strtolower($t), $valid_units)) {
            return true;
        }
    }
    return false;
}

function grab_base_ou($enum) {
    $base_dn = $enum['base_dn'];
    $base_ou = [];

    if ($base_dn) {
        $dn_parts = explode(',', $base_dn);
        foreach ($dn_parts as $element) {
            $parts = explode('=', $element);
            if (strtoupper($parts[0]) == 'OU') {
                $base_ou[] = $parts[1];
            }
        }
    }

    return (count($base_ou)) ? $base_ou : false;
}

function get_page_controls($all_checked, $current_page = 1, $page_count = 1, $page_size = 50, $search = "") {

    
    $back_btn_disabled = ($current_page == 1) ? 'disabled' : '';
    $forward_btn_disabled = ($current_page == $page_count) ? 'disabled' : '';
    
    $page_sizes = [50, 100, 250, 500, 1000];
    $page_size_options = '';
    foreach ($page_sizes as $option) {
        $selected = ($page_size == $option) ? 'selected' : '';
        $page_size_options .= '<option value="' . $option . '" ' . $selected . '>' . $option . ' ' . _('Per Page') . '</option>';
    }

    $close_button = '';
    if (!empty($search)) {
        $close_button = '
        <button id="ad-search-clear-btn" class="ui-btn ui-btn-outlined">
            <span class="material-symbols-outlined md-pointer md-20 md-400">close</span>
        </button>';
    }

    $all_checked_state = ($all_checked) ? 'checked' : '';
    $check_all_text = ($all_checked) ? _("Select None") : _("Select All");

    $html_page_controls = '
        <div class="form-inline tw-flex tw-flex-row tw-justify-between tw-m-5">
            <div class="tw-flex">
                <div class="tw-w-[200px]">
                    <input type="text" id="ad-search" class="ui-input" value="' . $search . '" placeholder="Search this folder...">
                </div>
                <button id="ad-search-btn" class="ui-btn ui-btn-outlined tw-ml-1" ><span class="material-symbols-outlined md-pointer md-20 md-400">search</span></button>
                ' . $close_button . '
            </div>
            <div class="ad-page-control-container tw-flex tw-justify-between  tw-w-72">
                <button class="ui-btn ui-btn-outlined ad-page-btn" ' . $back_btn_disabled . ' value="' . ($current_page - 1) . '" title="' .  _('Previous Page') . '">
                    <span class="material-symbols-outlined md-400 md-18">
                        keyboard_arrow_left
                    </span>
                </button>
                <span class="ad-pager tw-w-48 tw-flex tw-flex-row tw-border tw-items-center tw-justify-evenly tw-rounded-md tw-h-9 tw-px-2">
                    <span class="tw-pr-2">
                        ' . _('Page') . '
                    </span>
                    <span class="tw-flex tw-items-center">
                        <span class="tw-block tw-w-12">
                            <input type="text" id="ad-pager-page-number" class="ui-input tw-p-1" name="page" value="' . $current_page . '">
                        </span>
                        <span class="tw-px-2">/</span>
                        <span class="pagination-total">' . $page_count . '</span>
                    </span>
                </span>
                <button class="ui-btn ui-btn-outlined ad-page-btn" ' . $forward_btn_disabled . ' value="' . ($current_page + 1) . '" title="' .  _('Next Page') . '">
                    <span class="material-symbols-outlined md-400 md-18">
                        keyboard_arrow_right
                    </span>
                </button>
            </div>
            <div>
                <select name="ad-num-records" class="ui-select num-records">
                    ' . $page_size_options . '
                </select>
                <input type="hidden" id="current-page-size" value="'.$page_size.'"/>
            </div>
            <div class="tw-border tw-flex tw-items-center tw-rounded-md tw-mr-2 tw-p-1.5">
                <label class="checkbox">
                    <input type="checkbox" class="toggle-users ui-checkbox-group-item " '.$all_checked_state.'>
                    <span>'.$check_all_text.'</span>
                </label>
            </div>
        </div>';

    return $html_page_controls;
}
