<?php

define('BASEPATH', dirname(__FILE__));

include '/var/www/html/nagioslogserver/application/config/config.local.php';
$opensearch_password =  $config['opensearch_password'];

$dest_ip = 'localhost';

$short_options="s:b:e:i:clr";
$long_options=["source:","begindate:", "enddate:", "configuration", "logs", "indexes:", "restart"];

$options = getopt($short_options, $long_options);

syslog(LOG_INFO, "Nagios Log Server Migration: options " . json_encode($options));
$restart = false;

if (isset($options['r']) || isset($options['restart'])) {
    $restart = true;
}

if (strlen($options['s']) > 0) {
    $source_ip = $options['s'];
} else if (strlen($options['source']) > 0) {
    $source_ip = $options['source'];
}

if (!$source_ip) {
    echo "Source IP Address must be provided\n";
    exit(1);
}

if (((isset($options['i']) || isset($options['indexes']))) && 
    (isset($options['b']) || isset($options['e']) || isset($options['begindate']) || isset($options['enddate']))) {
    syslog(LOG_INFO,'Nagios Log Server Migration: Specify indexes or date range, not both');
    echo "Specify indexes or date range, not both\n";
    exit(1);
}

$begin_date = "0000.01.01";
$end_date = (new DateTime())->modify('+1 day')->format('Y.m.d');

$provided_indices = false;

if (!((isset($options['i']) || isset($options['indexes'])))) {
    if (isset($options['b'])) {
        $begin_date = $options['b'];
        $provided_indices = true;
    } else if (isset($options['begindate'])) {
        $begin_date = $options['begindate'];
        $provided_indices = true;
    }
    if (isset($options['e'])) {
        $end_date = $options['e'];
        $provided_indices = true;
    } else if (isset($options['enddate'])) {
        $end_date = $options['enddate'];
        $provided_indices = true;
    }

    // Ensure dates are valid dates.
    $begints = date_parse_from_format('Y.m.d',($begin_date));
    $endts = date_parse_from_format('Y.m.d',($end_date));
    if ($begints['warning_count'] > 0 || $begints['error_count'] > 0) {
        echo "begin date must be in the format YYYY.MM.DD\n";
        die;
    }

    if ($endts['warning_count'] > 0 || $endts['error_count'] > 0) {
        echo "end date must be in the format YYYY.MM.DD\n";
        die;
    }

    echo "Migrating log data from $begin_date to $end_date\n";
} else {
    $provided_indices = true;
}

$source_url_base = "http://$source_ip:9200";
$destination_url_base = "https://$dest_ip:9200";
$progress_index = "nagioslogserver_migration";

// Internal types to migrate
$migrate_types = [
    'nagioslogserver' =>
        [/*'index_usage', 'commands',*/ 'cf_option', /*'snapshot', 'node', 'query',*/ 'user', /*'history'*/],
    'nagioslogserver_log' =>
            [],
//        ['JOBS', 'MAINTENANCE', 'CONFIG', 'SECURITY', 'BACKUP', 'POLLER', 'INFO', 'ALERT'],
    'kibana-int' =>
            [],
//        ['dashboard', 'report'],
    'nagioslogserver_history' => 
            [],
//        ['alert'],
];

$ignore_cf_options = [
    'maintenance_settings',
    'ncpa_api_token',
    'license_key',
    'interface_url',
    'email_from',
    'backup_rotation',
    'is_installed',
    'ncpa_nrdp_token',
    'last_update_check',
    'last_met_check_time',
    'lmd',
    'activation_key'
];

$source_session = curl_init();
$dest_session = curl_init();

if (!$source_session || !$dest_session) {
    print "Curl Setup failed.\n";
    exit(1);
}


curl_setopt($dest_session, CURLOPT_RETURNTRANSFER, true);
curl_setopt($dest_session, CURLOPT_SSL_VERIFYHOST, false);
curl_setopt($dest_session, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($dest_session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);


curl_setopt($source_session, CURLOPT_RETURNTRANSFER, true);
curl_setopt($source_session, CURLOPT_SSL_VERIFYHOST, false);
curl_setopt($source_session, CURLOPT_SSL_VERIFYPEER, false);

// Extract Index Names from cluster:
curl_setopt($source_session, CURLOPT_URL, "$source_url_base/_cat/indices");

$response = curl_exec($source_session);

if (!$response) {
    print "Failed to get index list from source server.\n";
    exit(1);
}

$response = preg_split("/\r\n|\r|\n/", $response);
$response = preg_replace('/ +/', ' ', $response);

$nls_indexes = array();
$logstash_indexes = array();

$indexes = array();

// Get Index Data
foreach  ($response as $row) {
    if (strlen($row) == 0) {
        continue;
    }
    $index = make_index_info_from_row($row);
    $indexes[$index['name']] = $index;
    if ($index['status'] == 'open') {
        if (strpos($index['name'], "logstash") !== false) {
            if (!(isset($options['i']) || isset($options['indexes'])) && !(isset($options['c']) || isset($options['configuration']))) {
                if ($index['name'] >= "logstash-$begin_date" && $index['name'] <= "logstash-$end_date") {
                    $logstash_indexes[] = $index['name'];
                }
            }
        } else {
            $nls_indexes[] = $index['name'];
        }
    }
}

if (isset($options['i']) || isset($options['indexes'])) {
    if (isset($options['indexes'])) {
        $indexes_command = $options['indexes'];
    } else {
        $indexes_command = $options['i'];
    }
    $logstash_indexes = explode(',', $indexes_command);
    echo "Migrating data from the following indexes: $indexes_command\n";

}

if (!$provided_indices) {
    $logstash_indexes = [];
}

// Create the migration index
create_migration_index();

if (isset($options['c']) || isset($options['configuration'])) {
    syslog(LOG_INFO, 'Nagios Log Server Migration: Migrating Configuration');
    // Migrate the NLS Core Data
    foreach ($nls_indexes as $nls_index) {
        echo "Migrating Log Server Index: $nls_index\n";
        syslog(LOG_INFO, "Nagios Log Server Migration: Migating Log Server Index: $nls_index\n");
        migrate_nls_index($source_session, $dest_session, $nls_index);
    }
}

if (isset($options['l']) || isset($options['logs']) || isset($options['i']) || isset($options['index'])) {
    // Migrate the actual log data
    foreach ($logstash_indexes as $logstash_index) {
        print "Migrating Logstash Index: $logstash_index\n";
        // Ensure we have enough capaccity for migrating the index
        close_old_indexes_if_necessary();
        reindex_nls_index($logstash_index, $restart);
        // Ensure we have enough capaccity for ingesting more data
        close_old_indexes_if_necessary();
    }
}

function create_migration_index() {
    global $destination_url_base, $progress_index, $opensearch_password;

    $session = curl_init();
    curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);

    $request_body = '{
        "settings": {
          "index": {
            "number_of_shards": 1,
            "number_of_replicas": 1
          }
        },
        "mappings": {
          "properties": {
            "source_ip": {
              "type": "text"
          }
        }
    }';

    curl_setopt($session, CURLOPT_URL, "$destination_url_base/$progress_index");
    curl_setopt($session, CURLOPT_CUSTOMREQUEST, 'PUT');
    curl_setopt($session, CURLOPT_HTTPHEADER, array('Content-Type:application/json'));
    curl_setopt($session, CURLOPT_POSTFIELDS, $request_body);

    curl_exec($session);
}

function make_index_info_from_row($index_row) {
    $index_array = explode(' ', $index_row);
    $base_column = 1;
    if ($index_array[0] == 'close') {
        $base_column = 0;
    }

    $index_info = array();
    $index_info['name'] = $index_array[$base_column + 1];
    $index_info['status'] = $index_array[$base_column];
    if ($base_column == 1 && count(array_keys($index_array)) > 6)  {
        $index_info['primary'] = intval($index_array[$base_column + 2]);
        $index_info['replicas'] = intval($index_array[$base_column + 3]);
        $index_info['doc_count'] = intval($index_array[$base_column + 4]);
        $index_info['doc_deleted'] = intval($index_array[$base_column + 5]);
        $index_info['total_size'] = $index_array[$base_column + 6];
        $index_info['primary_size'] = $index_array[$base_column + 7];
    }

    return $index_info;    
}

function get_docs_per_minute($index) {
    $request_body = '{
        "aggs": {
            "total": {
                "date_histogram": {
                    "field": "@timestamp",
                    "interval": "1m"
                }
            }
        }
    }';

    global $source_url_base;

    $session = curl_init();
    curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($session, CURLOPT_URL, "$source_url_base/$index/_search?search_type=count");
    curl_setopt($session, CURLOPT_CUSTOMREQUEST, 'POST');
    curl_setopt($session, CURLOPT_HTTPHEADER, array('Content-Type:application/json'));
    curl_setopt($session, CURLOPT_POSTFIELDS, $request_body);

    $response = curl_exec($session);
    $response = json_decode($response, true);

    if (isset($response['aggregations']['total']['buckets'])) {
        return $response['aggregations']['total']['buckets'];
    }

    return $response;
}

function aggregate_groupings($index, $number_of_logs = 100000) {
    $doc_aggregates = get_docs_per_minute($index);

    $new_time_ranges = array();
    $doc_count = 0;
    $start_time = '';

    for ($i = 0; $i < count($doc_aggregates); $i++) {
        if ($doc_count >= $number_of_logs) {
            $end_time = $doc_aggregates[$i]['key_as_string'];
            $aggregation = array();
            $aggregation['from'] = $start_time;
            $aggregation['to'] = $end_time;
            $new_time_ranges[] = $aggregation;
            $doc_count = 0;
        }
        if ($doc_count == 0) {
            $start_time = $doc_aggregates[$i]['key_as_string'];
        }

        $doc_count += $doc_aggregates[$i]['doc_count'];
    }
    $aggregation = array();
    $aggregation['from'] = $start_time;
    $end_time = $doc_aggregates[count($doc_aggregates) - 1]['key'] / 1000 + 60 ;
    $aggregation['to'] = gmdate('Y-m-d\TH:i:s.v\Z', $end_time);
    $new_time_ranges[] = $aggregation;

    return $new_time_ranges;
}

function reindex_batch($index, $date_range) {
    global $destination_url_base, $source_url_base, $indexes, $opensearch_password;

    $start_timestamp = date("Y/M/d H:i:s");

    $reindex_script = "
    Pattern ipv4 = /([0-9]{1,3}\.){3}[0-9]{1,3}/;
    Pattern ipv6 = /(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:))/;
    def legacy_doc = ctx._source;
    def new_doc = new HashMap();
    new_doc.put('@timestamp', legacy_doc.remove('@timestamp'));
    new_doc.put('type', legacy_doc.remove('type'));
    def host = new HashMap();
    if (legacy_doc.host ==~ ipv4 || legacy_doc.host ==~ ipv6) {
        host.put('ip', legacy_doc.remove('host'));
    } else {
        host.put('name', legacy_doc.remove('host'));
    }
    new_doc.put('message', legacy_doc.remove('message'));
    if (new_doc.type == 'syslog') {
        def log = new HashMap();
        def syslog = new HashMap();
        def facility = new HashMap();
        facility.put('code', legacy_doc.remove('facility'));
        facility.put('name', legacy_doc.remove('facility_label'));
        syslog.put('facility', facility);
        def severity = new HashMap();
        severity.put('code', legacy_doc.remove('severity'));
        severity.put('name', legacy_doc.remove('severity_label'));
        syslog.put('severity', severity);
        syslog.put('priority', legacy_doc.remove('priority'));
        log.put('syslog', syslog);
        new_doc.put('log', log);
        def process = new HashMap();
        process.put('name', legacy_doc.remove('program'));
        process.put('pid', legacy_doc.remove('pid'));
        new_doc.put('process', process);
        host.put('hostname', legacy_doc.remove('logsource'));
        legacy_doc.remove('timestamp');
    }
    if (new_doc.type == 'eventlog') {
        host.put('hostname', legacy_doc.remove('Hostname'));
        def process = new HashMap();
        process.put('threadid', legacy_doc.remove('ThreadID'));
        process.put('pid', legacy_doc.remove('ProcessID'));
        new_doc.put('process', process);
        def event = new HashMap();
        event.put('id', legacy_doc.remove('EventID'));
        event.put('type', legacy_doc.remove('EventType'));
        event.put('provider', legacy_doc.remove('Channel'));
        def event_time = new HashMap();
        event_time.put('original', legacy_doc.remove('EventTime'));
        event_time.put('received', legacy_doc.remove('EventReceivedTime'));
        event.put('time', event_time);
        event.put('severity', legacy_doc.remove('SeverityValue'));
        event.put('severity_name', legacy_doc.remove('Severity'));
        new_doc.put('event', event);
        def user = new HashMap();
        user.put('domain', legacy_doc.remove('Domain'));
        user.put('name', legacy_doc.remove('AccountName'));
        user.put('id', legacy_doc.remove('UserID'));
        user.put('type', legacy_doc.remove('AccountType'));
        new_doc.put('user', user);
        def nxlog = new HashMap();
        def nxlogmod = new HashMap();
        nxlogmod.put('type', legacy_doc.remove('SourceModuleType'));
        nxlogmod.put('name', legacy_doc.remove('SourceModuleName'));
        nxlog.put('module', nxlogmod);
        def log = new HashMap();
        log.put('level', legacy_doc.remove('Opcode'));
        new_doc.put('log', log);
        new_doc.put('nxlog', nxlog);
    }
    if (new_doc.type == 'apache_access') {
        def http = new HashMap();
        def request = new HashMap();
        request.put('referer', legacy_doc.remove('referrer'));
        request.put('method', legacy_doc.remove('verb'));
        http.put('request', request);
        def response = new HashMap();
        response.put('bytes', legacy_doc.remove('bytes'));
        response.put('status_code', legacy_doc.remove('response'));
        http.put('response', response);
        http.put('version', legacy_doc.remove('httpversion'));
        new_doc.put('http', http);
        def client = new HashMap();
        client.put('ip', legacy_doc.remove('clientip'));
        client.put('ident', legacy_doc.remove('ident'));
        new_doc.put('client', client);
        def url = new HashMap();
        def original = new HashMap();
        url.put('original',  legacy_doc.remove('request'));
        new_doc.put('url', url);
        def user_agent = new HashMap();
        user_agent.put('name', legacy_doc.remove('agent'));
        new_doc.put('user_agent', user_agent);
        legacy_doc.remove('timestamp');
        def log = new HashMap();
        def syslog = new HashMap();
        def facility = new HashMap();
        facility.put('code', legacy_doc.remove('facility'));
        facility.put('name', legacy_doc.remove('facility_label'));
        syslog.put('facility', facility);
        def severity = new HashMap();
        severity.put('code', legacy_doc.remove('severity'));
        severity.put('name', legacy_doc.remove('severity_label'));
        syslog.put('severity', severity);
        syslog.put('priority', legacy_doc.remove('priority'));
        log.put('syslog', syslog);
        new_doc.put('log', log);
        def user = new HashMap();
        user.put('name', legacy_doc.remove('auth'));
        new_doc.put('user', user);
        legacy_doc.remove('program');
        legacy_doc.remove('logsource');
    }
    new_doc.put('host', host);
    new_doc.put('@version', legacy_doc.remove('@version'));
    if (legacy_doc.size() > 0) {
        new_doc.put('legacy_event', legacy_doc);
    }
    ctx._source = new_doc;
";

    $reindex_query = [];
    $reindex_query['source']['remote']['host'] = $source_url_base;
//    $reindex_query['source']['remote']['username'] = $username;
//    $reindex_query['source']['remote']['password'] = $password;
    $reindex_query['source']['index'] = $index;
    $reindex_query['source']['size'] = 10000;
    $reindex_query['source']['remote']['socket_timeout'] = "2m";
    $reindex_query['source']['remote']['connect_timeout'] = "2m";
    $reindex_query['dest']['index'] = $index;
    $reindex_query['source']['query']['filtered']['filter']['bool']['must'][0]['range']['@timestamp'] = $date_range;
    // Map old fields to new ECS based fields.
    $reindex_query['script']['inline'] = $reindex_script;
    $session = curl_init();
    curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($session, CURLOPT_URL, "$destination_url_base/_reindex");
    curl_setopt($session, CURLOPT_CUSTOMREQUEST, 'POST');
    curl_setopt($session, CURLOPT_HTTPHEADER, array('Content-Type:application/json'));
    curl_setopt($session, CURLOPT_POSTFIELDS, json_encode($reindex_query));
    curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);
    $response = curl_exec($session);
    echo json_encode($response) . "\n\n";
    $results = json_decode($response, true);
    $results['start_time'] = $start_timestamp;
    $results['finish_time'] = date("Y/M/d H:i:s");
    $indexes[$index]['transfer_results'][$date_range['from']]['range'] = $date_range;
    $indexes[$index]['transfer_results'][$date_range['from']]['results'] = $results;

    log_migration_results($index, $date_range, $results);

}

function remove_date_range_from_status_doc($index, $date_range) {
    $status_doc = load_progress_for_index($index);
    $index_num = find_index_of_date_range($status_doc, $date_range);
    if ($index_num === -1) {
        return;
    }

    array_splice($status_doc['log_range'], $index_num, 1);
    save_progress_for_index($index, $status_doc);
}

function split_batch_range($index, $date_range) {
    // Otherwise, split the range in half and reindex each one; eventually we'll get down to manageable sized data, or single failures.
    $from = strtotime($date_range['from']);
    $to = strtotime($date_range['to']);

    // TODO: Ensure this gets down to microseconds, using floating point math if necessary.
    $midpoint = gmdate('Y-m-d\TH:i:s.v\Z', ($from + $to) / 2);

    print "The previous migration of this range had problems. Dividing and conquering it.\n";
    remove_date_range_from_status_doc($index, $date_range);

    // Replace date range in status doc with "fake" statuses for the new ranges
    // in case the migration was interrupted.
    $fake_results = array();
    $fake_results[0]['range']['from'] = $date_range['from'];
    $fake_results[0]['range']['to'] = $midpoint;
    $fake_results[0]['results']['failures'][0] = $midpoint;
    $fake_results[1]['range']['from'] = $midpoint;
    $fake_results[1]['range']['to'] = $date_range['to'];
    $fake_results[1]['results']['failures'][0] = $midpoint;

    $progress = load_progress_for_index($index);
    $progress['log_range'] = array_merge($progress['log_range'], $fake_results);

    save_progress_for_index($index, $progress);

    return array(
        array('from' => $date_range['from'], 'to' => $midpoint),
        array('from' => $midpoint, 'to' => $date_range['to'])
    );
}

function reindex_nls_index($index, $restart)
{
    if ($restart) {
        // Delete the migration progress doc for this index
        delete_progress_for_index($index);
    }
    
    $date_ranges = aggregate_groupings($index);

    foreach($date_ranges as $date_range) {
        print "Migrating index $index from " . $date_range['from'] . " to " . $date_range['to'] . "\n";

        $progress = load_progress_for_index($index);
        $batch_index = find_index_of_date_range($progress, $date_range);
        if ($batch_index != -1) {
            $reindex_batches = array();
            $batch_status = $progress['log_range'][$batch_index];
            // If it was reindexed with no errors and no failures, we can ignore it, save for the TODO below.
            print json_encode($date_range) . "\n";
            if (!isset($batch_status['results']['error']) && count($batch_status['results']['failures']) === 0) {
                print "Not dividing and conquering\n";
                // TODO Check to see how many docs are in this date range on the source and the destination and migrate if necessary.
                // Check that the ranges match up:
                if ($batch_status['range']['to'] == $date_range['to']) {
                    print "No need to migrate this batch.\n";
                } else {
                    print "Need to get the subsequent ranges and re-attempt them\n";
                    for ($i = $batch_index; $i < count($progress['log_range']) && $progress['log_range'][$i]['range']['from'] < $date_range['to']; $i++) {
                        $test_range = $progress['log_range'][$i];
                        // Only do the ones that have errors.
                        if (isset($test_range['results']['error']) || count($test_range['results']['failures']) > 0) {
                            $new_batches = split_batch_range($index, $test_range['range']);
                            foreach($new_batches as $new_batch) {
                                $reindex_batches[] = $new_batch;
                            }
                        }
                    }
                }
            } else {
                // Otherwise, split the range in half and reindex each one; eventually we'll get down to manageable sized data, or single failures.
                $reindex_batches = split_batch_range($index, $date_range);
                print "Divided range into " . json_encode($reindex_batches) . "\n";
            }

            if(count($reindex_batches) > 0) {
                for ($i = 0; $i < count($reindex_batches); $i++) {
                    print "Re-Migrating index $index from " . $reindex_batches[$i]['from'] . " to " . $reindex_batches[$i]['to'] . "\n";
                    reindex_batch($index, array('from' => $reindex_batches[$i]['from'], 'to' => $reindex_batches[$i]['to']));
                }
            }

            continue;
        }

        reindex_batch($index, $date_range);
    }
}

function delete_progress_for_index($index) {
    global $destination_url_base, $progress_index, $opensearch_password, $source_ip;

    $url = "$destination_url_base/$progress_index/_doc/$source_ip-$index";

    $session = curl_init();
    curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($session, CURLOPT_URL, "$url");
    curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);
    curl_setopt($session, CURLOPT_CUSTOMREQUEST, 'DELETE');
    
    curl_exec($session);
}

function load_progress_for_index($index) {
    global $destination_url_base, $progress_index, $opensearch_password, $source_ip;

    $url = "$destination_url_base/$progress_index/_doc/$source_ip-$index";

    $session = curl_init();
    curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($session, CURLOPT_URL, "$url");
    curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);

    $response = curl_exec($session);

    $result = json_decode($response, true);

    if (isset($result['found']) && isset($result['_source'])) {
        return $result['_source'];
    }

    return [];
}

function save_progress_for_index($index, $status_doc) {
    global $destination_url_base, $progress_index, $opensearch_password, $source_ip;

    $url = "$destination_url_base/$progress_index/_doc/$source_ip-$index";

    uasort($status_doc['log_range'], 'log_status_array_sort_callback');
    $status_doc['log_range'] = array_values(($status_doc['log_range']));
    $status_doc['source_ip'] = $source_ip;

    $session = curl_init();
    curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($session, CURLOPT_URL, "$url");
    curl_setopt($session, CURLOPT_CUSTOMREQUEST, 'PUT');
    curl_setopt($session, CURLOPT_POSTFIELDS, json_encode($status_doc));
    curl_setopt($session, CURLOPT_HTTPHEADER, array('Content-Type:application/json'));
    curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);

    $response = curl_exec($session);

    return $response;
}

function find_index_of_date_range($status_doc, $date_range) {
    if (!isset($status_doc['log_range'])) {
        return -1;
    }

    for ($i = 0; $i < count($status_doc['log_range']); $i++) {
        if ($status_doc['log_range'][$i]['range']['from'] == $date_range['from']) {
            return $i;
        }
    }

    return -1;
}

function log_status_array_sort_callback($a, $b) {
    $retval = strcmp($a['range']['from'], $b['range']['from']);
    return $retval;
}

function log_migration_results($index, $date_range, $results) {
    $status_doc = load_progress_for_index($index);

    $index_of_status = find_index_of_date_range($status_doc, $date_range);
    // TODO: consider if migration was complete save for the error or failure.
    // look at results: total, updated, created, versus count of failures.
    if ($index_of_status == -1) {
        $result = array();
        $result['range']['from'] = $date_range['from'];
        $result['range']['to'] = $date_range['to'];
        $result['results'] = $results;
        $status_doc['log_range'][] = $result;
    } else {
        $status_doc['log_range'][$index_of_status]['range']= $date_range;
        $status_doc['log_range'][$index_of_status]['range'] = $date_range;
        $status_doc['log_range'][$index_of_status]['results'] = $results;    
    }

    save_progress_for_index($index, $status_doc);
}

function get_documents_from_index($curl_session, $index, $start = 0, $count = 100) {
    global $source_url_base;
    $query = '{
        "from": ' . $start . ',
        "size": ' . $count . '
      }';
    curl_setopt($curl_session, CURLOPT_URL, "$source_url_base/$index/_search");
    curl_setopt($curl_session, CURLOPT_CUSTOMREQUEST, 'POST');
    curl_setopt($curl_session, CURLOPT_POSTFIELDS, $query);

    return curl_exec($curl_session);
}

function make_jsonl_for_record($index, $record, $upsert = false) {
    $type = $record->_type;
    if ($index == 'kibana-int') {
        $dest_index = strtolower('nagioslogserver_' . $type . 's');
    } else {
        $dest_index = strtolower($index . '_' . $type);
    }
    $id = $record->_id;
    $record_data = $record->_source;
    $record_data->_type = $type;
    $data = json_encode($record_data);

    $operation = $upsert ? '"index":' : '"create":';

    $jsonl = "{" . $operation . "{ \"_index\": \"$dest_index\", \"_id\": \"$id\" } }\n";
    $jsonl .= " $data \n";

    return $jsonl;
}

function do_bulk_insert($session, $jsonl) {
    global $destination_url_base, $opensearch_password;

    curl_setopt($session, CURLOPT_URL, "$destination_url_base/_bulk");
    curl_setopt($session, CURLOPT_CUSTOMREQUEST, 'POST');
    curl_setopt($session, CURLOPT_POSTFIELDS, $jsonl);
    curl_setopt($session, CURLOPT_HTTPHEADER, array('Content-Type:application/x-ndjson'));
    curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);
    curl_setopt($session, CURLOPT_RETURNTRANSFER, true);

    $response = curl_exec($session);

    // TODO: Handle response ->"errors":true case
    //echo "Nagios Log Server Migration Bulk Insert Response: $response";
    //syslog(LOG_INFO, "Nagios Log Server Migration Bulk Insert Response: $response");
}

function migrate_nls_index($src, $dest, $index) {
    global $migrate_types, $ignore_cf_options;

    $results_per_query = 10000;
    $start = 0;
    $total_hits = 0;
    $types = array();
    $valid_types = $migrate_types[$index];

    while (true) {
        print "migrating from record $start ";
        syslog(LOG_INFO, "Nagios Log Server Migration: migrating from record $start ");
        $result = get_documents_from_index($src, $index, $start, $results_per_query);
        $response = json_decode($result, false);
        $hit_count = count($response->hits->hits);
        $start += $hit_count;
        $total_hits += $hit_count;

        $jsonl = "";
        foreach ($response->hits->hits as $hit) {
            // Ignore internal metadata.
            if (strpos($index, 'logstash') == false && is_array($valid_types) && !in_array($hit->_type, $valid_types)) {
                continue;
            }
            print "Migrating index $index hit type: " . $hit->_type . " id: " . $hit->_id . "\n";


            // Do not migrate the nagiosadmin user
            if ($index === 'nagioslogserver' && $hit->_type === 'user') {
                if ($hit->_id == 1 || $hit->_source->username_lower === 'nagiosadmin') {
                    continue;
                }

                $hit->_source->default_dashboard = 'default';
            }

            $upsert = ($index === 'nagioslogserver' && $hit->_type === 'cf_option' && !in_array($hit->_id, $ignore_cf_options));
            print "index: $index, type: " . $hit->_type . ", id: " . $hit->_id . ", upsert: $upsert\n";
            // TODO: Filter out existing dashboard and report id's so we don't try to recreate them
            $record = make_jsonl_for_record($index, $hit, $upsert);
            $jsonl .= $record;
        }

        print " to record $total_hits\n";

        do_bulk_insert($dest, $jsonl);

        if ($hit_count < $results_per_query) {
            break;
        }
    }
}

function get_open_indices() {
    global $opensearch_password, $destination_url_base;
    $session = curl_init();

    $urlq = array("size" => 10000);

    curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);

    curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($session, CURLOPT_HTTPHEADER, array('Accept: application/json'));

    $uri = "$destination_url_base/_cat/indices";
    // Extract Index Names from cluster:
    curl_setopt($session, CURLOPT_URL, $uri);
    return json_decode(curl_exec($session),true);
}

function count_opensearch_nodes() {
    global $opensearch_password, $destination_url_base;
    $session = curl_init();

    $urlq = array("size" => 10000);

    curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);

    curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);

    $uri = "$destination_url_base/nagioslogserver_node/_search?" . http_build_query($urlq);
    // Extract Index Names from cluster:
    curl_setopt($session, CURLOPT_URL, $uri);
    $r = json_decode(curl_exec($session),true);
    // Take into account the global node here. It doesn't count as a node in the cluster.
    return $r['hits']['total']['value'] - 1;
}

function count_opensearch_open_shards() {
    $indices = get_open_indices();

    $shards = 0;

    foreach($indices as $index) {
        if ($index['status'] == 'open') {
            $shards += $index['pri'] + $index['rep'];
        }
    }
    // Take into account the global node here. It doesn't count as a node in the cluster.
    return $shards;
}

function close_index($index_name) {
    global $opensearch_password, $destination_url_base;
    $session = curl_init();

    $urlq = array("size" => 10000);

    curl_setopt($session, CURLOPT_USERPWD, 'nagioslogserver:' . $opensearch_password);

    curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($session, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($session, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($session, CURLOPT_POST, true);

    $uri = "$destination_url_base/$index_name/_close";
    echo "Closing Index $index_name\n";
    // Extract Index Names from cluster:
    curl_setopt($session, CURLOPT_URL, $uri);
    curl_exec($session);
}

function close_oldest_index() {
    $indices = get_open_indices();
    $logstash_indices = [];

    $today = 'logstash-' . gmdate('Y.m.d');

    foreach($indices as $index) {
        if ($index['status'] == 'open' && 
            preg_match('/^logstash-\d\d\d\d\.\d\d\.\d\d/', $index['index']) &&
            $index !== "$today") {
            $logstash_indices[] = $index['index'];
        }
    }

    if (count($logstash_indices) > 0) {
        sort($logstash_indices);
        close_index($logstash_indices[0]);
    }
}

function close_old_indexes_if_necessary() {
    // The limit for total open shards is 1000 per node.
    $max_shards = 1000 * count_opensearch_nodes();
    $total_shards = count_opensearch_open_shards();

    echo "Total Open Shards: $total_shards, max shards: $max_shards\n";
    // Ensure the number of open shards is less than 90% of the max shards.
    while ($total_shards > $max_shards * 0.9) {
        close_oldest_index();
        $total_shards = count_opensearch_open_shards();
    }

}
