#!/bin/bash
#
# Bash script for restoring Nagios Log Server backups
# Copyright (c) 2014-2017 Nagios Enterprises, LLC. All rights reserved.
#
# Restores the backups created for the 3 internal Nagios Log Server databases
#

INDEXNAMES=( "nagioslogserver_snapshot" "nagioslogserver_scheduled_report" "nagioslogserver_history_alert" \
"nagioslogserver_query" "nagioslogserver_snmp_reactor" "nagioslogserver_alert" "nagioslogserver_node" \
"nagioslogserver_host_list" "nagioslogserver_user" "nagioslogserver_reactor_server" \
"nagioslogserver_log" "nagioslogserver_deleted_alert" "nagioslogserver_email_template" "nagioslogserver_commands" \
"nagioslogserver_dashboards" "nagioslogserver_nrdp_server" "nagioslogserver_migration" "nagioslogserver_reports" \
"nagioslogserver_snmp_receiver" "nagioslogserver_easy_buttons" "nagioslogserver_index_usage" \
"nagioslogserver_cf_option" )

LOGSERVER_DIR="/usr/local/nagioslogserver"
BACKUP_DIR="/store/backups/nagioslogserver"
PYTHON=$(which python 2>/dev/null || which python3 2>/dev/null || which python2 2>/dev/null)

if [ -z "$1" ];then
    echo "Backup file must be specified as the first argument"
    echo "e.g. ./$0 backup_name.tar.gz"
    exit 1
fi

# Restoring mapping files with the index mapping data
echo "Starting Nagios Log Server Restore"
echo "----------------------------------"

filename=$1
if [[ $filename != *.tar.gz ]]; then
    filename="$filename.tar.gz"
fi
folder=${filename:0:${#filename}-7}
echo "Backup folder: $folder"
echo ""

# Extract the file given and start the actual updating
echo "Extracting the backups."
cur_dir=`pwd`
cd $BACKUP_DIR
tar xf $filename
cd $folder

# Loop through all the indexes and restore them one by one
echo "Creating restore jobs for indexes."
for index in "${INDEXNAMES[@]}"; do
    echo "Restoring: $index "
    # Delete and restore the index
    if [ -f "$folder/$index" ]; then
        # Delete index
        curl -u nagioslogserver:$nlspass --cacert /usr/local/nagioslogserver/opensearch/config/root-ca.pem -H "Content-type:application/json" -XPOST "https://localhost:9200/$index/_delete_by_query?conflicts=proceed" -d '{"query":{"match_all":{}}}' > /dev/null 2>&1

        # Restore the index by importing the index file
        php $LOGSERVER_DIR/scripts/dump_index.php --output "https://localhost:9200" --input "$folder/$index" 2>/dev/null
    else
        printf "\n\n ERROR: Backup file $filename was not found\n"
        exit 1
    fi
    
    echo -n " done."

done
echo "All indexes restored."

# Reset backend jobs
echo "Resetting backend jobs... "
curl -u nagioslogserver:$nlspass --cacert /usr/local/nagioslogserver/opensearch/config/root-ca.pem -XPOST 'https://localhost:9200/nagioslogserver/commands/snapshots_maintenance/_update' -d '{ "doc": { "run_time": 0, "status": "waiting" } }' > /dev/null 2>&1
curl -u nagioslogserver:$nlspass --cacert /usr/local/nagioslogserver/opensearch/config/root-ca.pem -XPOST 'https://localhost:9200/nagioslogserver/commands/backups/_update' -d '{ "doc": { "run_time": 0, "status": "waiting" } }' > /dev/null 2>&1
curl -u nagioslogserver:$nlspass --cacert /usr/local/nagioslogserver/opensearch/config/root-ca.pem -XPOST 'https://localhost:9200/nagioslogserver/commands/run_update_check/_update' -d '{ "doc": { "run_time": 0, "status": "waiting" } }' > /dev/null 2>&1
curl -u nagioslogserver:$nlspass --cacert /usr/local/nagioslogserver/opensearch/config/root-ca.pem -XPOST 'https://localhost:9200/nagioslogserver/commands/run_all_checks/_update' -d '{ "doc": { "run_time": 0, "status": "waiting" } }' > /dev/null 2>&1
curl -u nagioslogserver:$nlspass --cacert /usr/local/nagioslogserver/opensearch/config/root-ca.pem -XPOST 'https://localhost:9200/nagioslogserver/commands/cleanup_cmdsubsys/_update' -d '{ "doc": { "run_time": 0, "status": "waiting" } }' > /dev/null 2>&1
echo "done."

# Restore snapshots
cd $LOGSERVER_DIR/opensearch/snapshots
tar zxvf $folder/snapshots.tar.gz
chown nagios:nagios -R $LOGSERVER_DIR/opensearch/snapshots

# Apply new configuration of ES
php /var/www/html/nagioslogserver/www/index.php configure apply_to_instances

# Clean up
rm -rf "$folder"

echo ""
echo "Restore Complete!"
