#!/bin/bash
# check_os_cluster_status.sh - Monitor a local OpenSearch cluster's health.

if command -v python3 &> /dev/null
then
    python_cmd="python3"
else
    echo "UNKNOWN: Python is not installed"
    exit 3
fi

if [ -f /usr/local/nagioslogserver/opensearch/config/opensearch_config.php ]; then
    nlspass=`grep opensearch_password /usr/local/nagioslogserver/opensearch/config/opensearch_config.php | sed "s/.* = '\(.*\)';/\1/"`
else
    nlspass=`grep opensearch_password /var/www/html/nagioslogserver/application/config/config.local.php | sed "s/.* = '\(.*\)';/\1/"`
fi

es_output=`curl -u nagioslogserver:$nlspass --cacert /usr/local/nagioslogserver/opensearch/config/root-ca.pem -s https://localhost:9200/_cluster/health | $python_cmd -c "
import sys, json
data = json.load(sys.stdin)
if 'number_of_nodes' in data and 'status' in data: sys.stdout.write('{} {}'.format(data['status'], data['number_of_nodes']))"`

nodes_output=`curl  -H 'Content-Type: application/json' -XGET -u nagioslogserver:$nlspass --cacert /usr/local/nagioslogserver/opensearch/config/root-ca.pem -s https://localhost:9200/nagioslogserver_node/_search -d '{"size":10000,"query":{"bool":{"must_not":[{"match":{"_id":"global"}}]}}}' | $python_cmd -c "
import sys, json
data = json.load(sys.stdin)
if 'hits' in data and 'total' in data['hits']: sys.stdout.write('{}'.format(data['hits']['total']['value']))"`

if [ -z "$es_output" ]; then
	echo "UNKNOWN: could not retrieve cluster health data"
	exit 3
fi

# If the cluster status is GREEN or RED, set OK or CRITICAL (respectively) and exit immediately.
# If the cluster status is YELLOW, check the number of instances. 1-instance clusters are always yellow, so return OK in this case.
for i in $es_output; do
	case $i in

		"green")
			echo "OK: Cluster status is GREEN"
			exit 0
			;;
		"red")
			echo "CRITICAL: Cluster status is RED"
			exit 2
			;;
		"yellow")
			# Check the number of instances in this cluster.
			if [ $nodes_output -ge 2 ]; then
				echo "WARNING: Cluster status is YELLOW"
				exit 1
			fi
			;;
		"1")
			echo "OK: Cluster status is YELLOW but there is only one instance in the cluster"
			exit 0
			;;
		*)
			echo "WARNING: Cluster status is YELLOW"
			exit 1
			;;
	esac
done
