#!/bin/bash
# Checks the current percent usage of the OpenSearch JVM heap for all nodes.

if command -v python3 &> /dev/null
then
    python_cmd="python3"
else
    echo "UNKNOWN: Python is not installed"
    exit 3
fi


WARNING="100"
WARNING="0 0 100"
CRITICAL="100"
CRITICAL="0 0 100"

# Outputs a parsed threshold in the form "inverted low high" as a string.
# inverted will be 1 if @ was present, 0 otherwise.
# low and high will be integers on either side of the ":".
function parse_threshold() {
	raw=$1
	raw=$(echo $raw | sed 's/%//g')

	check_inverse=$(echo $raw | sed 's/@//')
	if [ "x$check_inverse" != "x$raw" ]; then
		# @ was present in threshold - low and high represent the "alerting" range
		echo -n "1 "
	else
		# threshold normal - low and high represent the "acceptable" range
		echo -n "0 "
	fi

	check_range=$(echo $check_inverse | sed 's/:$/:100/g' | sed 's/^:/0:/g' | sed 's/:/ /')
	if [ "x$check_range" != "x$check_inverse" ]; then
		# : was present in threshold - low and high both exist
		for term in $check_range; do
			echo -n "$term "
		done
	else
		echo -n "0 $check_range "
		# Only the high value is shown in the threshold
	fi
}

# Outputs 1 if the alert was triggered, 0 otherwise.
# call like $(check_threshold $threshold $value)
function check_threshold() {
	inverted=$1
	low=$2
	high=$3
	val=$4

	if [ $inverted -eq 1 ]; then
		if [ $val -ge $low ] && [ $val -le $high ]; then
			echo "1"
		else 
			echo "0"
		fi
	else
		if [ $val -lt $low ] || [ $val -gt $high ]; then
			echo "1"
		else
			echo "0"
		fi
	fi
}


while getopts "h?w:c:" opt; do

	case "$opt" in

		h|\?)
			echo "Usage: ./$0 [-h] [-w WARNING_PERCENT] [-c CRITICAL_PERCENT]"
			echo "This plugin reports based on the maximum heap usage found in your OpenSearch cluster. -w and -c values should be expressed as integers."
			echo "e.g. ./$0 -w 50 -c 70"
			echo "If a threshold is not set, it defaults to '100', i.e. it should never alert."
			exit 0
			;;
		w)
			WARNING_RAW=$OPTARG
			WARNING=$(parse_threshold $OPTARG)
			;;
		c)
			CRITICAL_RAW=$OPTARG
			CRITICAL=$(parse_threshold $OPTARG)
			;;
	esac
done

if [ -f /usr/local/nagioslogserver/opensearch/config/opensearch_config.php ]; then
    nlspass=`grep opensearch_password /usr/local/nagioslogserver/opensearch/config/opensearch_config.php | sed "s/.* = '\(.*\)';/\1/"`
else
    nlspass=`grep opensearch_password /var/www/html/nagioslogserver/application/config/config.local.php | sed "s/.* = '\(.*\)';/\1/"`
fi

jvm_stats=`curl -u nagioslogserver:$nlspass --cacert /usr/local/nagioslogserver/opensearch/config/root-ca.pem -s -XGET 'https://localhost:9200/_nodes/stats/jvm?pretty' | $python_cmd -c "
import sys,json
data = json.load(sys.stdin)
if 'nodes' not in data or not data['nodes']:
	sys.exit(3)
print(' '.join([str(data['nodes'][node]['jvm']['mem']['heap_used_percent']) for node in data['nodes']]))
"`

HIGH_WATER_MARK=0
HWM_COUNTER=0
MAX_HEAP_USED_PERCENT=0;
for x in $jvm_stats; do

	DO_WARN=$(check_threshold $WARNING $x)

	if [ $DO_WARN -eq 1 ]; then
		if [ $HIGH_WATER_MARK -lt 1 ]; then
			HWM_COUNTER=1
			HIGH_WATER_MARK=1
		elif [ $HIGH_WATER_MARK -eq 1 ]; then
			HWM_COUNTER=$((HWM_COUNTER + 1))
		fi
	fi

	DO_CRIT=$(check_threshold $CRITICAL $x)

	if [ $DO_CRIT -eq 1 ]; then
		if [ $HIGH_WATER_MARK -lt 2 ]; then
			HWM_COUNTER=1
			HIGH_WATER_MARK=2
		elif [ $HIGH_WATER_MARK -eq 2 ]; then
			HWM_COUNTER=$((HWM_COUNTER + 1))
		fi
	fi

	if [ $MAX_HEAP_USED_PERCENT -lt $x ]; then
		MAX_HEAP_USED_PERCENT=$x
	fi
done

case $HIGH_WATER_MARK in 
	0)
		OUTPUT_KEYWORD="OK"
		;;
	1)
		OUTPUT_KEYWORD="WARNING"
		;;
	2)
		OUTPUT_KEYWORD="CRITICAL"
		;;
	*)
		OUTPUT_KEYWORD="UNKNOWN"
		;;
esac

echo "$OUTPUT_KEYWORD: Maximum heap_used_percent is $MAX_HEAP_USED_PERCENT | max_heap_used_percent=$MAX_HEAP_USED_PERCENT%;$WARNING_RAW;$CRITICAL_RAW;0;100"
exit $HIGH_WATER_MARK