#
# Dev Environment Setup Script
#
# Before running set projectdir to the path of the nagiosna folder.
#
# Sets up a Laravel scheduling cron job and Supervisor for Laravel Reverb and queue workers in a dev environment.
# This script overwrites the cron job and Supervisor .ini files created by the fullinstall script.
#
# Useful Supervisor commands:
# View running processes: sudo supervisorctl status
# Start or stop processes: sudo supervisorctl <start/stop> <all/process name>
#

projectdir="/repos/nagios-analyzer-next/nagiosna"

# Create cron job for Laravel scheduler
echo "Setting up Laravel scheduler cron job..."
PHP_PATH="$(which php)"
CRON_JOB="* * * * * cd $projectdir && $PHP_PATH artisan schedule:run >> $projectdir/storage/logs/cron.log 2>&1"
crontab -u root - <<EOF
$CRON_JOB
EOF
echo "Crontab for root set to: $CRON_JOB"

# Setup Supervisor for Laravel queues and Reverb
SUPERVISOR_INI_DIR="/etc/supervisord.d"
LOG_FILE_DIR="$projectdir/storage/logs"

# systemctl enable supervisord

# Workers use queue:listen instead of queue:work for the dev environment
# queue:listen does not require restarting when changes are made to the code.
echo "Creating laravel-worker.ini..."
sudo tee "$SUPERVISOR_INI_DIR/laravel-worker.ini" > /dev/null <<EOF
[program:laravel-worker]
process_name=%(program_name)s_%(process_num)02d
command=$PHP_PATH $projectdir/artisan queue:listen --timeout=0 
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=root
numprocs=6
redirect_stderr=true
stdout_logfile=$LOG_FILE_DIR/worker.log
stopwaitsecs=86400
startsecs=1
EOF

echo "Creating laravel-reverb.ini..."
sudo tee "$SUPERVISOR_INI_DIR/laravel-reverb.ini" > /dev/null <<EOF
[program:laravel-reverb]
command=$PHP_PATH $projectdir/artisan reverb:start --debug
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=root
redirect_stderr=true
stdout_logfile=$LOG_FILE_DIR/reverb.log
EOF

if ! pgrep -x "supervisord" > /dev/null; then
  echo "Starting supervisord..."
  sudo systemctl enable supervisord
  sudo systemctl start supervisord
else
  echo "supervisord is already running. Restarting..."
  sudo supervisorctl stop all
fi

echo "Reloading Supervisor config..."
sudo supervisorctl reread
sudo supervisorctl update

echo "Starting new processes..."
sudo supervisorctl start all
sudo supervisorctl status

echo "Supervisor setup completed!"