#
# libinstall.sh - common functions for installing Nagios software
#

# To use, simply put this file into the same directory as your install file.
# Then add this to the top of your install script:
#     cd $(dirname $(readlink -e "$0"))
#     . ./libinstall.sh
# (The cd command ensures your script is run from the install directory so the
# library can be found, and readlink ensures this works even with symlinks.)
# This library will then automatically run some standard checks for you like
# making sure it's run as root, making sure RHEL systems are registered, etc.
# It will also detect various information about the OS and make that available
# in global variables (see the set_os_info function for the list of variables).
#
# Write your installation steps either as functions or separate scripts. Then
# create a fullinstall function that puts it all together by passing those
# steps to the run_steps wrapper function, which will take care of some error
# handling and progress output. Finally, pass your fullinstall function to the
# log_it wrapper function, which will log the output of the installer to the
# specified file.

# Exit immediately if an untested command fails
set -e

# Global environment variable that determines whether to prompt the user for various information or not
export INTERACTIVE="True"

BLUE='\033[38;2;41;180;255m'
RED='\033[38;2;255;0;0m'
NC='\033[0m' # Reset

# Prints with a color: echo_c "text" $COLOR
echo_c() {
    local text="$1"
    local color="$2"
    printf "${color}%s${NC}\n" "$text"
}

# Wrapper function for running installation steps
run_steps() {
	local traps
	traps=$(trap)

	for step; do
		# Set trap to print error msg if step fails
		trap "step_error $step" 0

		echo_c "Running '$step'..." "$BLUE"

		if [ -f "installed.$step" ] || [ -f "installed.all" ]; then
			echo_c "$step step already completed - skipping" "$RED"

			# Reset traps
			eval "$traps"

			continue
		fi

		"$step"

		echo "$step step completed OK"
		touch "installed.$step"
	done

	# Reset traps
	eval "$traps"

	touch installed.all
	for step; do
		rm -f "installed.$step"
	done
}

create_install_log() {
	# Initialize install.log
	log="install.log"

	cat >>"$log" <<-EOF
		Nagios Network Analyzer Installation Log
		==========================
		DATE: $(date)

		DISTRO INFO:
		$distro
		$version
		$architecture

	EOF

	# Redirect all stdout and stderr to log file and remove ANSI colors while still showing on terminal
	exec > >(tee >(sed 's/\x1B\[[0-9;]*[JKmsu]//g' >> "$log")) 2>&1
}

# Gets run automatically when an installation step fails
step_error() {
	cat >&2 <<-EOF

		===================
		INSTALLATION ERROR!
		===================
		Installation step failed - exiting.
		Check for error messages in the install log (install.log).

		If you require assistance in resolving the issue, please include install.log
		in your communications with Nagios Enterprises technical support.

		The step that failed was: '$1'
	EOF
}

# Wrapper function for capturing and logging stdout & stderr of another command
# First argument is the log file, followed by the command to run
log_it() {
	log="$1"
	shift
	"$@" 2>&1 | tee -a "$log"
}

# Print installation header text. Takes one argument - the name of the product.
print_header() {
	local product
	product="$1"

	cat <<-EOF

		$product Installation
		$(echo $product Installation | sed 's/./=/g')
		DATE: $(date)
	
		DISTRO INFO:
		$distro
		$version
		$architecture

	EOF
}

# Print installation footer text. Takes two arguments - the name of the product
# and optionally the web server path to the web interface.
print_footer() {
	local product path
	product="$1"
	path="$2"

	echo
	echo "$product Installation Success!"
	echo

	get_ip

	if [ -n "$path" ]; then
		echo "You can finish the final setup steps for $product by visiting:"
		echo "    http://$ip/$path/"
		echo
	fi

	
}

# Checks whether the product (or any other Nagios product) is installed
# and whether this distribution of Linux is supported for the current version of the product.
do_install_check() {
	# Check if any other product is already installed
	products=( nagioslogserver nagiosxi nagiosfusion )
	for dir in "{$products[@]}"; do
		if [ -d "/usr/local/$dir" ]; then
			echo "It looks like another product ($dir) is already installed on this system." >&2
			exit 1
		fi
	done

    if [ "$dist" == "el8" ] && [ "$distro" == "CentOS" ]; then
        echo "" >&2
        echo "CentOS 8 is not currently supported. Please use one of the following distros:" >&2
        echo "  CentOS 9 or 10" >&2
        echo "  RHEL, Oracle 9" >&2
        echo "  Ubuntu 24.04 LTS" >&2
        echo "  Debian 12 or 13" >&2
        echo "" >&2
		exit 1
	elif [ "$dist" == "el8" ] || [ "$dist" == "el9" ] || [ "$dist" == "el10" ]; then
		echo "Installing on $distro $version"
	elif [ "$dist" == "ubuntu22" ] || [ "$dist" == "ubuntu24" ]; then
		echo "Installing on $distro $version"
	elif [ "$dist" == "debian12" ] || [ "$dist" == "debian13" ]; then
		echo "Installing on $distro $version"
	else
		echo "" >&2
		echo "$dist is not currently supported. Please use one of the following distros:" >&2
        echo "  CentOS 9 or 10" >&2
        echo "  RHEL, Oracle 9" >&2
        echo "  Ubuntu 24.04 LTS" >&2
        echo "  Debian 12 or 13" >&2
        echo "" >&2
		exit 1
	fi
}

# Adds specified user if it doesn't exist already
add_user() {
	local user
	user="$1"

	if ! grep -q "^$user:" /etc/passwd; then
		case "$dist" in
			el* )
				useradd -m "$user"
				;;
			* )
				useradd "$user"
		esac
	fi

    # Add user home folder if it doesn't already exist
    if [ ! -d "/home/$user" ]; then
        mkdir "/home/$user"
        chown $user:$user "/home/$user"
    fi
}

# Adds specified group if it doesn't exist already
add_group() {
	local group
	group="$1"

	if ! grep -q "^$group:" /etc/group; then
		groupadd "$group"
	fi
}

# Adds user to the specified groups
add_to_groups() {
	local user
	user="$1"

	shift
	for group; do
		usermod -a -G "$group" "$user"
	done
}

# Detect OS & set global variables for other commands to use.
# OS variables have a detailed long variable, and a "more useful" short one:
# distro/dist, version/ver, architecture/arch. If in doubt, use the short one.
set_os_info() {
	if [ `uname -s` != "Linux" ]; then
		error "Unsupported OS detected. Can currently only detects" \
			"Linux distributions."
	fi

	# Get OS & version
	if which lsb_release &>/dev/null; then
		distro=`lsb_release -si`
		version=`lsb_release -sr`
	elif [ -r /etc/redhat-release ]; then

		if rpm -q centos-linux-release || rpm -q centos-stream-release || rpm -q centos-release; then
			distro=CentOS
		elif [ -r /etc/oracle-release ]; then
			distro=OracleServer
		elif rpm -q redhat-release || rpm -q redhat-release-server || rpm -q sles_es-release-server || rpm -q sles_es-release; then
			distro=RedHatEnterpriseServer
		fi >/dev/null

		version=`sed 's/.*release \([0-9.]\+\).*/\1/' /etc/redhat-release`
	else
		# Release is not RedHat or CentOS, let's start by checking for SuSE
		# or we can just make the last-ditch effort to find out the OS by sourcing os-release if it exists
		if [ -r /etc/os-release ]; then
			source /etc/os-release
			if [ -n "$NAME" ]; then
				distro=$NAME
				version=$VERSION_ID
			fi
		fi
	fi

	# Add patch for CentOS Stream with lsb installed
	if [ "$distro" == "CentOSStream" ]; then
		distro="CentOS"
	fi
	
	# Add patch for RHEL with lsb installed
	if [ "$distro" == "RedHatEnterprise" ]; then
		distro=RedHatEnterpriseServer
	fi

	# Add patch for Debian which changed NAME on us...
	if [[ $distro == *"Debian"* ]]; then
		distro="Debian"
	fi

	# Add patch level to the version of SLES (because they don't...)
	if [ "$distro" == "SUSE LINUX" ]; then
		if [ -r /etc/SuSE-release ]; then
			patchlevel=$(cat /etc/SuSE-release | cut -d ' ' -f 3 -s | sed -n 3p)
			version="$version.$patchlevel"
		fi
	fi

	# Verify that we have a distro now
	if [ -z "$distro" ]; then
		echo "ERROR: Could not determine OS. Please make sure lsb_release is installed or your OS info is in /etc/os-release." >&2
		exit 1
	fi

	ver="${version%%.*}"

	case "$distro" in
		CentOS | RedHatEnterpriseServer | OracleServer | CloudLinux )
			dist="el$ver"
			;;
		Debian )
			dist="debian$ver"
			;;
		* )
			dist=$(echo "$distro$ver" | tr A-Z a-z)
	esac

	# TODO Clean up variables below. We arnt going to support all of the distros and may not use everything
	architecture=`uname -m`

	arch="$architecture"
	httpd='httpd'
	mysqld='mysqld'

	apacheuser='apache'
	apachegroup='apache'
	nagiosuser='nagios'
	nagiosgroup='nagios'
	nagioscmdgroup='nagcmd'

	phpini='/etc/php.ini'
	phpconfd='/etc/php.d'
	httpdconfdir='/etc/httpd/conf.d'
	mrtgcfg='/etc/mrtg/mrtg.cfg'
	mibsdir='/usr/share/snmp/mibs'

	# This is set later by php-config if possible
	php_extension_dir='/usr/lib64/php/modules'
	# TODO update the below for when we set up install for other distros
	case "$dist" in
        el8 | el9 )
                mysqld="mariadb"
                ntpd="chronyd"
			;;
		ubuntu22 | ubuntu24 | debian11 | debian12 | debian13 )
			mysqld="mysql"
			mibsdir="/usr/share/mibs"
            ntpd="ntp"
            if [ "$dist" == "ubuntu24" ]; then
                ntpd="ntpsec"
                phpini="/etc/php/8.3/apache2/php.ini"
                phpconfd="/etc/php/8.3/apache2/conf.d"
                phpconfdcli="/etc/php/8.3/cli/conf.d"
                phpcliini="/etc/php/8.3/cli/php.ini"
                mibsdir="/usr/share/snmp/mibs"
            elif [ "$dist" == "ubuntu22" ]; then
                phpini="/etc/php/8.1/apache2/php.ini"
                phpconfd="/etc/php/8.1/apache2/conf.d"
                phpconfdcli="/etc/php/8.1/cli/conf.d"
                phpcliini="/etc/php/8.1/cli/php.ini"
                mibsdir="/usr/share/snmp/mibs"
			elif [ "$dist" == "debian13" ]; then
				ntpd="ntpsec"
            	mysqld="mariadb"
            	phpini="/etc/php/8.4/apache2/php.ini"
                phpconfd="/etc/php/8.4/apache2/conf.d"
                phpconfdcli="/etc/php/8.4/cli/conf.d"
                phpcliini="/etc/php/8.4/cli/php.ini"
                mibsdir="/usr/share/snmp/mibs"
			elif [ "$dist" == "debian12" ]; then
				ntpd="ntpsec"
            	mysqld="mariadb"
            	phpini="/etc/php/8.2/apache2/php.ini"
                phpconfd="/etc/php/8.2/apache2/conf.d"
                phpconfdcli="/etc/php/8.2/cli/conf.d"
                phpcliini="/etc/php/8.2/cli/php.ini"
                mibsdir="/usr/share/snmp/mibs"
            elif [ "$dist" == "debian11" ]; then
                mysqld="mariadb"
                phpini="/etc/php/7.4/apache2/php.ini"
                phpconfd="/etc/php/7.4/apache2/conf.d"
                phpconfdcli="/etc/php/7.4/cli/conf.d"
                phpcliini="/etc/php/7.4/cli/php.ini"
                mibsdir="/usr/share/snmp/mibs"
            else
                phpini="/etc/php5/apache2/php.ini"
                phpconfd="/etc/php5/apache2/conf.d"
                phpconfdcli="/etc/php5/cli/conf.d"
                phpcliini="/etc/php5/cli/php.ini"
            fi
			httpdconfdir="/etc/apache2/sites-available"
            apacheuser="www-data"
            apachegroup="www-data"
          	httpdconf="/etc/apache2/apache2.conf"
            httpdroot="/var/www/html"
			httpd="apache2"
            crond="cron"
			;;
	esac
}

# Open the specified TCP ports
open_tcp_ports() {
	local chain rulenum

	if [ "$dist" == "el7" ] || [ "$dist" == "el8" ] || [ "$dist" == "el9" ] || [ "$dist" == "el10" ]; then
		if [ `command -v firewalld` ]; then
			set +e
			for port; do
				firewall-cmd --zone=public --add-port="${port/:/-}"/tcp --permanent
			done
			firewall-cmd --reload
			set -e
		fi
	else
		# determine information for the rules
		chain=$(iptables -L | awk '/^Chain.*INPUT/ {print $2; exit(0)}')
		rulenum=$((`iptables -L $chain | wc -l` - 2))

		# test to make sure we aren't using less than the minimum 1
		if [ $rulenum -lt 1 ]; then rulenum=1; fi

		# add the rules
		for port; do
			iptables -I "$chain" "$rulenum" -m state --state NEW -m tcp \
				-p tcp --dport "$port" -j ACCEPT
		done

		# save changes
		service iptables save
		service iptables restart
	fi
}

# Disable SELinux on RHEL-based systems
disable_selinux() {
	if selinuxenabled; then
		setenforce 0
		cat >/etc/selinux/config <<-EOF
			# This file controls the state of SELinux on the system.
			# SELINUX= can take one of these three values:
			#     enforcing - SELinux security policy is enforced.
			#     permissive - SELinux prints warnings instead of enforcing.
			#     disabled - No SELinux policy is loaded.
			SELINUX=disabled
			# SELINUXTYPE= can take one of these two values:
			#     targeted - Targeted processes are protected,
			#     mls - Multi Level Security protection.
			SELINUXTYPE=targeted
			# SETLOCALDEFS= Check local definition changes
			SETLOCALDEFS=0
		EOF
	fi
}

# Install SourceGuardian PHP extension
install_sourceguardian() {
	local phpver ixedfile entry zipfile

	# Get PHP version
	phpver=$(php -v | head -n 1 | cut -d ' ' -f 2 | cut -d . -f 1,2)

	ixedfile="ixed.$phpver.lin"
	entry="extension=$ixedfile"

	if [ "$arch" = "x86_64" ]; then
		zipfile="sourceguardian/ixed4.lin.x86-64.zip"
	else
		echo "Error: Cannot install on 32bit systems"
		exit 1
	fi


    # Make sure we are using the exact php extension dir
	if [ `command -v php-config` ]; then
		php_extension_dir=$(php-config --extension-dir)
	fi

	# Extract SourceGuardian extension to the proper directory
	unzip -o "$zipfile" "$ixedfile" -d "$php_extension_dir"

	if [ -f "$php_extension_dir/$ixedfile" ]; then
		echo "Sourceguardian extension found for PHP version $phpver"
	else
		error "No valid Sourceguardian extension found for PHP" \
			"version $phpver"
	fi

	if grep -q "$entry" "$phpini" "$phpconfd"/*; then
		echo "Sourceguardian extension already in php.ini"
	else
		echo "Adding Sourceguardian extension to php.ini"
		echo "$entry" > "$phpconfd/sourceguardian.ini"
	fi

	# Add to the CLI ini file if it exists (debian systems)
	if [ $phpcliini ]; then
		if grep -q "$entry" "$phpcliini"; then
			echo "Sourceguardian extension already in php.ini"
		else
			echo "Adding entry to PHP CLI php.ini file"
			echo "$entry" > "$phpcliini"
		fi
	fi
}

path_is_ok() {
	echo "$PATH" \
	| awk 'BEGIN{RS=":"} {p[$0]++} END{if (p["/sbin"] && p["/usr/sbin"]) exit(0); exit(1)}'
}

set_os_info