<?php

namespace App\Jobs;

use App\Models\WiresharkScan;
use App\Events\PacketCaptured;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Symfony\Component\Process\Process;
use Throwable;
use Illuminate\Support\Facades\Log;

class CapturePacketsJob implements ShouldQueue
{
    use Queueable;

    public int $scanId;
    public string $interface;
    public string $file;
    public string $timezone;
    public int $duration;
    public $timeout = 0;
    public $tries = 1;

    public function __construct(int $scanId, string $interface, string $file, string $timezone, int $duration)
    {
        $this->scanId = $scanId;
        $this->interface = $interface;
        $this->file = $file;
        $this->timezone = $timezone;
        $this->duration = $duration;
    }

    public function handle(): void
    {
        date_default_timezone_set($this->timezone);
        $scan = WiresharkScan::find($this->scanId);
        if (!$scan) {
            Log::warning("WiresharkScan not found for scan ID {$this->scanId}");
            return;
        }

        Log::info("Starting tshark", ['interface' => $this->interface, 'file' => $this->file,'duration' => $this->duration]);
        try {
            $command = [
                'sudo', '-u', 'nna',
                'tshark',
                '-i', $this->interface,
                '-w', $this->file,
                '-a', "duration:{$this->duration}",
            ];

            $process = new Process($command, '/var/www/html/nagiosna');
            $process->setTimeout(null);
            $process->start();

            $pid = $process->getPid();
            $scan->update([
                'status' => 'in_progress',
                'pid' => $pid,
            ]);

            Log::info("Started tshark with PID {$pid}");
            PacketCaptured::dispatch($scan);

            // Keep the job alive while the process is running
            $process->wait();

            if (file_exists($this->file)) {
                chmod($this->file, 0660); 
            }

            Log::info('TShark debug', [
                'exit_code' => $process->getExitCode(),
                'stdout' => $process->getOutput(),
                'stderr' => $process->getErrorOutput(),
            ]);

            $scan->refresh();

            // Update status and clear PID when process ends
            $scan->update([
                'status' => $process->isSuccessful() ? 'completed' : 'failed',
                'finished_at' => now('UTC'),
                'pid' => null,
            ]);

            Log::info('Tshark process finished', [
                'scan_id' => $this->scanId,
                'success' => $process->isSuccessful(),
            ]);

            PacketCaptured::dispatch($scan);

        } catch (Throwable $e) {
            Log::error('Wireshark scan failed to start or run', [
                'scan_id' => $this->scanId,
                'exception' => $e->getMessage(),
            ]);

            $scan->update([
                'status' => 'failed',
                'finished_at' => now('UTC'),
                'pid' => null,
            ]);

            PacketCaptured::dispatch($scan);
        }
    }

    public function failed(?Throwable $exception): void
    {
        $scan = WiresharkScan::find($this->scanId);
        if ($scan) {
             try {
                posix_kill($scan->pid, SIGTERM);
                Log::info("Killed tshark process with PID {$scan->pid} due to job failure.");
            } catch (\Throwable $e) {
                Log::error("Failed to kill tshark process PID {$scan->pid}: " . $e->getMessage());
            }
            $scan->update([
                'status' => 'failed',
                'finished_at' => now('UTC'),
                'pid' => null,
            ]);
            PacketCaptured::dispatch($scan);
        }

        Log::error('Wireshark scan job failed', [
            'scan_id' => $this->scanId,
            'error' => $exception?->getMessage(),
        ]);
    }
}
