<?php

namespace App\Jobs;

use App\Models\Ndiff;
use App\Models\NmapScan;
use App\Events\NdiffUpdate;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Log;
use Symfony\Component\Process\Process;
use Throwable;

class NdiffJob implements ShouldQueue
{
    use Queueable;

    public $tries = 1;

    /**
     * Middleware to ensure no other job with the same key runs until this one finishes.
     */
    public function middleware()
    {
        return [new \Illuminate\Queue\Middleware\WithoutOverlapping($this->ndiff->id)];
    }

    /**
     * Create a new job instance.
     */
    public function __construct(
        public Ndiff $ndiff,
        public NmapScan $nmapScan1,
        public NmapScan $nmapScan2
    ) {}

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $outputFilename = "ndiff_" . $this->nmapScan1->output_filename . "_vs_" . $this->nmapScan2->output_filename;
        $outputFilepath = '/usr/local/nmap/ndiff/' . $outputFilename;
        $xmlFilepath = $outputFilepath . '.xml';
        $txtFilepath = $outputFilepath. '.txt';

        $scan1Filepath = '/usr/local/nmap/' . $this->nmapScan1->output_filename . '.xml';
        $scan2Filepath = '/usr/local/nmap/' . $this->nmapScan2->output_filename . '.xml';

        if (!file_exists($scan1Filepath) || !file_exists($scan2Filepath)) {
            Log::error('NdiffJob - scan file missing', [
                'ndiff_id' => $this->ndiff->id,
                'scan1Filepath' => $scan1Filepath,
                'scan2Filepath' => $scan2Filepath,
                'exists1' => file_exists($scan1Filepath),
                'exists2' => file_exists($scan2Filepath),
            ]);

            $this->ndiff->update([
                'status' => 'Failed',
                'finished_at' => now('UTC'),
            ]);
            NdiffUpdate::dispatch($this->ndiff);

            return;
        }

        $ndiffPath = trim(shell_exec('which ndiff'));
        if (!$ndiffPath || !file_exists($ndiffPath)) {
            Log::error('Ndiff executable not found');
            $this->ndiff->update([
                'status' => 'Failed',
                'finished_at' => now('UTC'),
            ]);
            NdiffUpdate::dispatch($this->ndiff);
            return;
        }

        // Log::info('Starting Ndiff', [
        //     'ndiff_id' =>  $this->ndiff->id, 
        //     'scan1_id' => $this->nmapScan1->id, 
        //     'scan2_id' => $this->nmapScan2->id,
        //     'xmlFilepath' => $xmlFilepath,
        //     'txtFilepath' => $txtFilepath,
        //     'scan1Filepath' => $scan1Filepath,
        //     'scan2Filepath' => $scan2Filepath
        // ]);

        $startTime = now('UTC');
        Log::info('NdiffJob - started at', ['ndiff_id' => $this->ndiff->id, 'started_at' => $startTime]);
        $this->ndiff->update([
            'status' => 'In Progress',
            'started_at' => $startTime,
            'output_filename' => $outputFilename,
        ]);
        NdiffUpdate::dispatch($this->ndiff);

        // Text Ndiff
        $command = [
            $ndiffPath,
            '--text',
            $scan1Filepath,
            $scan2Filepath,
        ];
        $process = new Process($command, '/var/www/html/nagiosna');
        $process->setTimeout(null);
        $process->run();

        if (!$process->isSuccessful() && $process->getExitCode() !== 1) {
            Log::error('NdiffJob - text failed', [
                'error' => $process->getErrorOutput(),
                'output' => $process->getOutput(),
            ]);

            $this->ndiff->update([
                'status' => 'Failed',
                'finished_at' => now('UTC'),
            ]);
            NdiffUpdate::dispatch($this->ndiff);

            return;
        }

        file_put_contents($txtFilepath, $process->getOutput());

        // XML Ndiff
        $command = [
            $ndiffPath,
            '--xml',
            $scan1Filepath,
            $scan2Filepath,
        ];
        $process = new Process($command, '/tmp');
        $process->setTimeout(null);
        $process->run();

        if (!$process->isSuccessful() && $process->getExitCode() !== 1) {
            Log::error('Ndiff XML failed', [
                'error' => $process->getErrorOutput(),
                'output' => $process->getOutput(),
            ]);

            $this->ndiff->update([
                'status' => 'Failed',
                'finished_at' => now('UTC'),
            ]);
            NdiffUpdate::dispatch($this->ndiff);
            
            return;
        }

        file_put_contents($xmlFilepath, $process->getOutput());

        $json = $this->createNdiffJson($outputFilepath);

        Log::info('Ndiff completed successfully', [
            'ndiff_id' => $this->ndiff->id,
        ]);

        $this->ndiff->update([
            'status' => 'Completed',
            'finished_at' => now('UTC'),
        ]);

        NdiffUpdate::dispatch($this->ndiff);
    }

    /**
     * Handle a job failure.
     */
    public function failed(?Throwable $exception)
    {
        $this->ndiff->update([
            'status' => 'Failed',
            'finished_at' => now('UTC'),
        ]);
        NdiffUpdate::dispatch($this->ndiff);
        Log::error("Ndiff job failed: " . $exception->getMessage());
    }

    /**
     * Create a JSON file from the XML output file.
     * 
     * @return array Associative array of JSON objects.
     */
    private function createNdiffJson(string $outputFilepath): array
    {
        $xmlFilepath = $outputFilepath . '.xml';
        $xml = simplexml_load_file($xmlFilepath);
        $rawJson = json_decode(json_encode($xml), true);
        $json = $this->promoteAttributes($rawJson);

        $jsonFilepath = $outputFilepath . '.json';
        if (file_exists($jsonFilepath)) {
            unlink($jsonFilepath);
        }
        file_put_contents($jsonFilepath, json_encode($json, JSON_PRETTY_PRINT));

        return $json;
    }

    /**
     * Recursively removes all '@attribute' keys caused by the XML to JSON conversion from the given array.
     */
    private function promoteAttributes(array $input): array 
    {
        foreach ($input as $key => $value) {
            if (is_array($value)) {
                $input[$key] = $this->promoteAttributes($value);
    
                // If '@attributes' exists, promote its keys
                if (isset($value['@attributes']) && is_array($value['@attributes'])) {
                    $input[$key] = array_merge($value['@attributes'], $input[$key]);
                    unset($input[$key]['@attributes']);
                }
            }
        }
    
        return $input;
    }
}