<?php

namespace App\Jobs;

use App\Models\NmapScan;
use App\Models\Ndiff;
use App\Events\NmapScanUpdate;
use App\Events\NdiffCreated;
use App\Jobs\NdiffJob;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\Middleware\WithoutOverlapping;
use Symfony\Component\Process\Process;
use Illuminate\Support\Facades\Log;
use Throwable;

class NmapScanJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $tries = 1;
    public $timeout = 0;

    /**
     * Middleware to ensure no other job with the same key runs until this one finishes.
     */
    public function middleware()
    {
        return [new WithoutOverlapping($this->nmapScan->id)];
    }

    /**
     * Create a new job instance.
     */
    public function __construct(
        public NmapScan $nmapScan
    ) {}

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        // Reload the model from the database to ensure we have a fresh instance
        $this->nmapScan = NmapScan::find($this->nmapScan->id);
        if (!$this->nmapScan) {
            Log::error('NmapScan not found in database', ['scan_id' => $this->nmapScan->id]);
            return;
        }

        if ($this->nmapScan->status !== 'Pending') {
            Log::warning('NmapScanJob attempted to run for scan not pending', [
                'scan_id' => $this->nmapScan->id,
                'status' => $this->nmapScan->status,
            ]);
            return;
        }

        $startTime = now('UTC');

        $profile = $this->nmapScan->profile;
        if ($profile) {
            $profile->increment('times_ran', 1);
        }

        $scheduledScan = $this->nmapScan->scheduledScan;
        if ($scheduledScan) {
            $scheduledScan->update([
                'last_run_at' => $startTime,
                'times_ran' => $scheduledScan->times_ran + 1,
            ]);
        }

        $scanParameters = $this->nmapScan->scan_parameters;
        
        $sanitizedTitle = preg_replace('/[^A-Za-z0-9_-]/', '_', $this->nmapScan->title);
        $outputFilename = $sanitizedTitle . '_' . $startTime->format('Y-m-d_H-i-s');
        $outputFilepath = '/usr/local/nmap/' . $outputFilename;

        $xmlFilepath = $outputFilepath . '.xml';
        $txtFilepath = $outputFilepath . '.txt';
        $logFilepath = $outputFilepath . '.log';

        Log::info('Starting Nmap Scan', ['scan_id' => $this->nmapScan->id, 'scan_parameters' => $scanParameters, 'output_filename' => $outputFilename, 'started_at' => $startTime,]);

        $command = sprintf(
            'sudo -u nna nmap --privileged --stats-every 5s -oX %s -oN %s %s > %s 2>&1',
            escapeshellarg($xmlFilepath),
            escapeshellarg($txtFilepath),
            $scanParameters,
            escapeshellarg($logFilepath)
        );

        $process = Process::fromShellCommandline($command);
        $process->setWorkingDirectory('/var/www/html/nagiosna');
        $process->setTimeout(null);
        $process->start();

        $this->nmapScan->update([
            'status' => 'In Progress',
            'started_at' => $startTime,
            'output_filename' => $outputFilename,
        ]);
        NmapScanUpdate::dispatch($this->nmapScan, '', 0.0, '');

        $handle = fopen($logFilepath, 'c+');
        stream_set_blocking($handle, false);
        $progressBuffer = '';
        while ($process->isRunning() || !feof($handle)) {
            $chunk = fread($handle, 4096);

            if ($chunk === false && $chunk === '') {
                continue;
            }

            $progressBuffer .= $chunk;
            while (($pos = strpos($progressBuffer, "\n")) !== false) {
                $line = trim(substr($progressBuffer, 0, $pos));
                $progressBuffer = substr($progressBuffer, $pos + 1);

                if ($line === '') {
                    continue;
                }
                // Log::info('Nmap progress', ['line' => $line]);
                if (preg_match('/^(\w.*?Scan) Timing: About ([\d.]+)% done; ETC: [^()]*\(([^)]+) remaining\)/', $line, $matches)) {
                    $scanType = trim($matches[1]);
                    $percent = round((float)$matches[2], 2);
                    $remainingTime = $matches[3];
                    NmapScanUpdate::dispatch($this->nmapScan, $scanType, $percent, $remainingTime);
                }
            }
            usleep(100000);
        }
        fclose($handle);

        if (file_exists($logFilepath)) {
            @unlink($logFilepath);
        }

        Log::info('Nmap Scan process result', ['scan_id' => $this->nmapScan->id, 'successful' => $process->isSuccessful(),]);
        if (!$process->isRunning() && !$process->isSuccessful()) {
            $this->nmapScan->refresh();

            if ($this->nmapScan->status === "Stopped") {
                return;
            }

            $this->nmapScan->update([
                'status' => 'Failed',
                'finished_at' => now('UTC'),
            ]);
            NmapScanUpdate::dispatch($this->nmapScan, '', 100.0, '');

            Log::error('Nmap scan process failed', ['scan_id' => $this->nmapScan->id, 'error' => $process->getErrorOutput()]);
            return;
        }

        $json = $this->createNmapScanJson($outputFilepath);

        $this->nmapScan->update([
            'status' => 'Completed',
            'finished_at' => now('UTC'),
        ]);
        NmapScanUpdate::dispatch($this->nmapScan, '', 100.0, '');

        // If this scan has a scheduled scan make an Ndiff
        $scheduledScan = $this->nmapScan->scheduledScan;
        if ($scheduledScan) {
            $previousScan = $scheduledScan->previousScan($this->nmapScan);

            if ($previousScan) {
                Log::info('Nmap Scan part of scheduled scan - creating Ndiff', ['scan_id' => $this->nmapScan->id, 'prev_scan_id' => $previousScan->id, ]);
                
                $ndiff = Ndiff::create([
                    'user_id'    => $this->nmapScan->user_id,
                    'scan1_id'   => $previousScan->id,
                    'scan2_id'   => $this->nmapScan->id,
                    'status'     => 'Pending',
                    'title'      => $scheduledScan->name . ' Comparison',
                ]);
        
                NdiffJob::dispatch($ndiff, $previousScan, $this->nmapScan);
                NdiffCreated::dispatch($ndiff);
            }
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(?Throwable $exception)
    {
        if (is_null($this->nmapScan->started_at)) {
            $this->nmapScan->started_at = now('UTC');
        }

        $this->nmapScan->update([
            'status' => 'Failed',
            'started_at' => $this->nmapScan->started_at,
            'finished_at' => now('UTC'),
        ]);
        NmapScanUpdate::dispatch($this->nmapScan, '', 100.0, '');
        Log::error("Nmap scan job failed: " . $exception->getMessage());
    }

    /**
     * Create a JSON file from the XML output file.
     * 
     * @return array Associative array of JSON objects.
     */
    private function createNmapScanJson(string $outputFilepath): array
    {
        $xmlFilepath = $outputFilepath . '.xml';

        if (!file_exists($xmlFilepath)) {
            Log::error("Nmap XML missing", [
                'scan_id' => $this->nmapScan->id,
                'xml_path' => $xmlFilepath,
            ]);
            throw new \Exception("Nmap XML file not found: {$xmlFilepath}");
        }

        $xml = simplexml_load_file($xmlFilepath);
        $rawJson = json_decode(json_encode($xml), true);
        $json = $this->promoteAttributes($rawJson);

        $jsonFilepath = $outputFilepath . '.json';
        if (file_exists($jsonFilepath)) {
            unlink($jsonFilepath);
        }
        file_put_contents($jsonFilepath, json_encode($json, JSON_PRETTY_PRINT));

        return $json;
    }

    /**
     * Recursively removes all '@attribute' keys caused by the XML to JSON conversion from the given array.
     */
    private function promoteAttributes(array $input): array 
    {
        foreach ($input as $key => $value) {
            if (is_array($value)) {
                $input[$key] = $this->promoteAttributes($value);
    
                // If '@attributes' exists, promote its keys
                if (isset($value['@attributes']) && is_array($value['@attributes'])) {
                    $input[$key] = array_merge($value['@attributes'], $input[$key]);
                    unset($input[$key]['@attributes']);
                }
            }
        }
    
        return $input;
    }
}
