<?php

namespace App\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use App\Models\SuricataConfig;
use App\Models\SuricataAlert;
use App\Models\SuricataScan;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;
use Symfony\Component\Yaml\Yaml;
use Illuminate\Support\Facades\Process;
use Illuminate\Support\Carbon;
use Throwable;
use App\Jobs\CheckJob;
use App\Models\Check;

class ProcessEveLogAlertsJob implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new job instance.
     */
    public function __construct()
    {
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        // checks the currently configured eve_log from suricata.yaml
        $suricataConfig = SuricataConfig::first();
        $suricataScan = SuricataScan::first();
        $suricataYaml = rtrim($suricataConfig->suricata_yaml_path, '/') . "/suricata.yaml";
        $yamlData = Yaml::parseFile($suricataYaml);
        $eveLogs = $yamlData['outputs'] ?? [];
        $eveLog = "eve.json";
        foreach ($eveLogs as $output) {
            if (isset($output['eve-log']) && isset($output['eve-log']['filename'])) {
                $eveLog = $output['eve-log']['filename'];
                break;
            }
        }
        $eveLog = rtrim($suricataConfig->suricata_log_path, '/') . "/" . $eveLog;
        Log::info("Reading {$eveLog} for alerts.");
        $expect_eve = $suricataScan->current_eve_log;
        $lastOffset = $suricataScan->eve_current_line;
        // If the eve log changes name assume its a new file completely 
        if ($expect_eve != $eveLog || $expect_eve == null){
            $suricataScan->current_eve_log = $eveLog;
            $lastOffset = 0;
        }
        
        if (!file_exists($eveLog)) {
            Log::warning("Eve log file not found at {$eveLog}");
            return;
        }

        $fp = fopen($eveLog, 'r');
        if (!$fp) {
            Log::error("Unable to open eve log file: {$eveLog}");
            return;
        }
        if (filesize($eveLog) < $lastOffset) {
            $lastOffset = 0;
            $suricataScan->last_rotated = now('UTC');
        }
        fseek($fp, $lastOffset);
        // begins to read the eve_log file from where it last left off 
        $counters = [];
        Log::info("Beggining readings for suricata alerts");
        while ($line = fgets($fp)) {
            $alertData = json_decode($line, true);
            if (!$alertData || ($alertData['event_type'] ?? '') !== 'alert') {
                continue;
            }
            $sigId = $alertData['alert']['signature_id'] ?? null;
            if (!$sigId) continue;

            $counters[$sigId]['timestamp'] = Carbon::parse($alertData['timestamp']) ?? now('UTC');
            $counters[$sigId]['count'] = ($counters[$sigId]['count'] ?? 0) + 1;
            $counters[$sigId]['name'] = $alertData['alert']['signature'] ?? null;
            $counters[$sigId]['category'] = $alertData['alert']['category'] ?? null;
            $counters[$sigId]['severity'] = $alertData['alert']['severity'] ?? null;
        }
        // save alerts to db
        foreach ($counters as $sigId => $data) {
            $alert = SuricataAlert::where('signature_id', $sigId)
                ->whereDate('created_at', now('UTC')->toDateString())
                ->first();

            if (!$alert) {
                $alert = SuricataAlert::create([
                    'signature_id'   => $sigId,
                    'signature_name' => $data['name'],
                    'category' => (isset($data['category']) && $data['category'] !== '') 
                        ? $data['category'] 
                        : "Uncategorized",
                    'severity'       => $data['severity'],
                    'last_hour_count'=> $data['count'],
                    'last_seen'      => $data['timestamp'],
                ]);
            } else {
                $alert->last_seen = $data['timestamp'];
                $alert->incrementLastHour($data['count']);
            }
        }
        // Save file offset to database and save the new file name from above if it occured
        Log::info("Saving offset in database");
        $suricataScan->eve_current_line = ftell($fp);
        $suricataScan->save();
        fclose($fp);

        // look if user has suricata checks defined and dispatch them if so
        $suricata_checks = Check::where('check_type', 'suricata')->where('active', 1)->get();
        Log::debug("Suricata checks: " . json_encode($suricata_checks));
        foreach ($suricata_checks as $check) {
            CheckJob::dispatch($check);
        }
    }


    public function failed(?Throwable $exception)
    {
        Log::error("Failed to read alerts from suricata: " . ($exception?->getMessage() ?? "No exception passed"));
    }
}
