<?php

namespace App\Jobs;

use App\Models\Report;
use App\Models\User;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\Middleware\WithoutOverlapping;
use Illuminate\Support\Facades\Log;
use App\Services\ExportService;
use App\Models\NotificationOptions;
use App\Notifications\Alert;
use App\Services\NotificationService;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Storage;
use Throwable;

class ReportJob implements ShouldQueue
{
  use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

  public $tries = 1;
  public $timeout = 0;

  /**
   * Middleware to ensure no other job with the same key runs until this one finishes.
   */
  public function middleware()
  {
      return [new WithoutOverlapping($this->report->id)];
  }

  /**
   * Create a new job instance.
   */
  public function __construct(
    public Report $report
  ) {}

  /**
   * Execute the job.
   */
  public function handle(): void
  {
    {
      Log::info("Running report job " . $this->report->name);
      
      $user = $this->report->user;
      if (!$user) {
        Log::warning("Report has no associated user", [
            'report_id' => $this->report->id
        ]);
        return;
      }

      $apikey = $user->apikey;

      if (!$apikey) {
        Log::warning("User has no API key", [
          'user_id' => $user->id,
          'report_id' => $this->report->id
        ]);
        return;
      }

      $result = ExportService::export(
        $this->report->dashboard_id,
        $apikey,
        $this->report->format
      );

      Log::info("Export finished", [
        'report' => $this->report->id,
        'user' => $user->id,
        'file' => $result['filename'] ?? 'no filename'
      ]);

      $notification_options = NotificationOptions::first();
      if (!$notification_options) {
        Log::error("Report [ " . $this->report->name . " (id: " . $this->report->id . ") ] no notification options found.");
        return;
      }
      $notification_options->alert_type = "report";
      NotificationService::configure($notification_options, $user);
      $notification_options->mail_message = (new MailMessage)
        ->mailer($notification_options->mail_config['mailer'])
        ->greeting('New Report!')
        ->attach('/var/www/html/nagiosna/storage/app/public/' . $result['filename']);

      $users = $this->report->users()->get();
      foreach ($users as $user) {
        try {
          $user->notify(new Alert($notification_options));
        } catch (\Throwable $e) {
          Log::warning("Failed to notify user ID {$user->id} ({$user->email}): " . $e->getMessage());
        }
      }

      Storage::disk('public')->delete($result['filename']);
    }
  }
}
