<?php

namespace App\Jobs;

use App\Models\SuricataRuleset;
use App\Models\SuricataConfig;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Process;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;
use Throwable;


class RulesetUpdateJob implements ShouldQueue
{
    use Queueable;

    protected string $frequency;
    /**
     * Create a new job instance.
     */
    public function __construct(string $frequency)
    {
        $this->frequency = $frequency;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        // Ensure umask allows group-write
        umask(0002);

        // Get all suricata rulesets that match the jobs update frequency and that are enabled
        $suricataNeedsRestart = false;
        $suricataConfig = SuricataConfig::first();
        if (!$suricataConfig) {
            Log::error("No SuricataConfig found.");
            return;
        }
        $rulesets = SuricataRuleset::where('update_frequency', $this->frequency)
            ->where('enabled', true)
            ->get();
        $rulesetNull = false;
        // Log if no rulesets are found or all the found enabled rulesets
        if ($rulesets->isEmpty()) {
            $rulesetNull = true;
            \Log::info("No rulesets found for frequency: {$this->frequency}");
        } else {
            \Log::info("Updating rulesets for frequency: {$this->frequency}");
            foreach ($rulesets as $ruleset) {
                \Log::info("→ Found ruleset: {$ruleset->name}");
            }
        }

        $suricataBinPath = "/usr/local/bin/suricata-update";

        // Create process with a cwd path
        $process = Process::path("/var/www/html/nagiosna");

        // Run the command to get all the enabled sources in order to turn them off before pulling new
        $enabledSourcesCmd = "sudo -u nna {$suricataBinPath} list-enabled-sources";
        \Log::info("Suricata find enabled sourced cmd", ['cmd' => $enabledSourcesCmd]);
        $result = $process->run($enabledSourcesCmd);

        // Log if theres an issue with the command
        if ($result->failed()) {
            \Log::error('Failed to list enabled Suricata sources: ' . $result->errorOutput());
            return;
        }

        $output = $result->output();

        // Collect all the names of the enabled sources ex: "- et/pro"
        preg_match_all('/^\s*-\s*(.+)$/m', $output, $matches);
        $enabledSources = $matches[1] ?? [];
        // Disable all the enabled sources
        foreach ($enabledSources as $source) {
            \Log::info("Disabling Suricata source: {$source}");

            $disable = $process->run("sudo -u nna {$suricataBinPath} disable-source {$source}");

            // Ensure the disable source runs
            if ($disable->failed()) {
                \Log::warning("Failed to disable source: {$source} → " . $disable->errorOutput());
            }
        }

        $rulesPath = rtrim($suricataConfig->suricata_rules_path, '/');
        $frequencies = ['5min', 'Hourly', 'Daily', 'Weekly', 'Monthly', 'Custom'];
        $currentOutputPath = "{$rulesPath}/{$this->frequency}";
        $suricataRulesFile = "{$rulesPath}/suricata.rules";
        $suricataRulesHeader = "{$rulesPath}/header.rules";

        // 1. Ensure output directories exist
        foreach ($frequencies as $folder) {
            $fullPath = "{$rulesPath}/{$folder}";
            if (!is_dir($fullPath)) {
                File::makeDirectory($fullPath, 02775, true);
                \Log::info("Created missing directory: {$fullPath}");
            }
        }

        $ruleFiles = glob("{$currentOutputPath}/*.rules");
        foreach ($ruleFiles as $file) {
            if (!unlink($file)) {
                \Log::warning("Failed to delete rules file: {$file}");
            }
        }
        \Log::info("Removed all rule files in: {$currentOutputPath}");

        // Create the unique header to ensure repeat rules don't exist
        if (!File::exists($suricataRulesHeader)) {
            \Log::info("Creating header.rules");
            $fetchCmd = "sudo -u nna {$suricataBinPath} -o {$currentOutputPath}";
            \Log::info("Running command to fetch rules for header.rules: {$fetchCmd}");
            $makeHeaderRulesFetch = $process->run($fetchCmd);

            if ($makeHeaderRulesFetch->failed()){
                \Log::error("Fetching rules for header.rules failed: " . $makeHeaderRulesFetch->errorOutput());
            } else {
                $moveCmd = "head -n 426 {$currentOutputPath}/suricata.rules > {$suricataRulesHeader}";
                \Log::info("Running command to create header.rules: {$moveCmd}");
                $makeHeaderRulesMove = $process->run($moveCmd);
                if ($makeHeaderRulesMove->failed()) {
                    \Log::error("Creating header.rules failed: " . $makeHeaderRulesMove->errorOutput());
                }
            }
        }

        // 2. Enable and update each ruleset individually
        foreach ($rulesets as $ruleset) {
            $name = str_replace("/", "_", $ruleset->name);
            // Enable
            $command = "{$suricataBinPath} enable-source {$ruleset->name}";
            if ($ruleset->secret_code_required) {
                $command .= " secret-code={$ruleset->secret_code}";
            }

            \Log::info("Enabling source: {$ruleset->name}");
            $enable = $process->run($command);

            if ($enable->failed()) {
                $ruleset->fetch_status = 'Failed';
                \Log::warning("Failed to enable source: {$ruleset->name} -> " . $enable->errorOutput());
                continue;
            }

            // Run suricata-update for this single source to its tmp file
            $update = $process->run("sudo -u nna {$suricataBinPath} -o {$currentOutputPath}");

            $output = $update->errorOutput();

            if (str_contains($output, '<Error>')) {
                $ruleset->fetch_status = 'Failed';
                \Log::error("Fetch failed for {$ruleset->name}: " . $output);
            } else {
                // put the actual unique rules in its own file
                $process->run("tail -n +427 {$currentOutputPath}/suricata.rules > {$currentOutputPath}/{$name}.rules");
                $process->run("rm -f {$currentOutputPath}/suricata.rules");
                $ruleset->fetch_status = 'Success';
                $suricataNeedsRestart = true;
                \Log::info("Fetched ruleset successfully: {$ruleset->name}");
                // need to check other directories to ensure the same ruleset isn't duplicated if its frequency is moved
                foreach($frequencies as $folder) {
                    if ($folder === $this->frequency || $folder === "Custom") {
                        continue;
                    }
                    $otherPath = "{$rulesPath}/{$folder}/{$name}.rules";
                    if (file_exists($otherPath)) {
                        unlink($otherPath);
                        \Log::info("Removed duplicate ruleset {$otherPath} from folder '{$folder}' due to frequency change.");
                    }
                }
            }

            $process->run("sudo -u nna {$suricataBinPath} disable-source {$ruleset->name}");
            // Update timestamp regardless of fetch result
            $ruleset->last_updated = now('UTC');
            $ruleset->save();
        }

        // 3. Check other ruleset folders to delete any rulesets that have been marked as disabled 
        foreach ($frequencies as $frequency) {
            if ($frequency == $this->frequency){
                continue;
            }
            $ruleFiles = glob("{$rulesPath}/{$frequency}/*.rules");

            foreach ($ruleFiles as $file) {
                $filenameNoExt = pathinfo($file, PATHINFO_FILENAME);

                $dbName = str_replace('_', '/', $filenameNoExt);
                \Log::info("dbName is {$dbName}");

                $ruleset = SuricataRuleset::where('name', $dbName)
                    ->where('update_frequency', $frequency)
                    ->first(['enabled']);
                \Log::info("Ruleset from dbname is: {$ruleset}");

                if (!$ruleset || !$ruleset->enabled) {
                    unlink($file);
                    $suricataNeedsRestart = true;
                    \Log::info("Deleted disabled or untracked ruleset file: {$file}");
                }
            }
        }

        // 4. Combine all frequency rules into final suricata.rules
        if (file_exists("{$currentOutputPath}/suricata.rules")) {
            unlink("{$currentOutputPath}/suricata.rules");
        }
        if ($rulesetNull) {
            $frequencies = array_diff($frequencies, [$this->frequency]); //remove the current frequency from the array
        }
        // Add the header to the top of the suricata.rules file
        $headerContnet = File::get($suricataRulesHeader);
        File::put($suricataRulesFile, $headerContnet);
        \Log::info("Added header to suricata.rules: {$suricataRulesHeader}");
        foreach ($frequencies as $folder) {
            $rulesPathPattern = "{$rulesPath}/{$folder}/*.rules";
            $ruleFiles = glob($rulesPathPattern);
            \Log::info($ruleFiles);
            if (!empty($ruleFiles)) {
                foreach ($ruleFiles as $file) {
                    $content = File::get($file);
                    File::append($suricataRulesFile, $content);
                    \Log::info("Appended {$file} to {$suricataRulesFile}");
                }
            } else {
                \Log::warning("No .rules files found in: {$rulesPathPattern}");
            }
        }

        // 5. Reload Suricata if running
        $pidResult = $process->run('pidof suricata');
        if ($pidResult->successful() && $suricataNeedsRestart) {
            $pid = trim($pidResult->output());
            \Log::info("Suricata running (PID: {$pid}) — sending USR2");
            $killResult = $process->run("kill -USR2 {$pid}");

            if ($killResult->failed()) {
                \Log::error("Failed to reload Suricata (PID: {$pid}): " . $killResult->errorOutput());
            } else {
                \Log::info("Successfully sent USR2 to Suricata.");
            }
        } else {
            \Log::warning("Suricata not running — no reload sent.");
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(?Throwable $exception)
    {
        $rulesets = SuricataRuleset::where('update_frequency', $this->frequency)
            ->where('enabled', true)
            ->get();
        if ($rulesets) {
            foreach ($rulesets as $ruleset){
                $ruleset->fetch_status = 'Failed';
                $ruleset->last_updated = now('UTC');
                $ruleset->save();
            }
        }
        Log::error("Suricata ruleset update failed: " . $exception->getMessage());
    }
}
