<?php

namespace App\Jobs;

use App\Models\SuricataConfig;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Process;
use App\Events\SuricataPcapScanUpdate;

class SuricataReadPcapJob implements ShouldQueue
{
    use Queueable;

    protected string $pcapPath;
    protected string $outputFileName;
    protected int $deletePcap;

    /**
     * Create a new job instance.
     */
    public function __construct(string $pcapPath, string $outputFileName, int $deletePcap)
    {
        $this->pcapPath = $pcapPath;
        $this->outputFileName = $outputFileName;
        $this->deletePcap = $deletePcap;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $suricata = SuricataConfig::first();

        $logBasePath = rtrim($suricata->suricata_log_path, '/');
        $suricataYamlPath = rtrim($suricata->suricata_yaml_path, '/');

        $outputFileName = pathinfo($this->outputFileName, PATHINFO_FILENAME) . '.json';
        $outputFilePath = $logBasePath . '/' . $outputFileName;

        $suricataBin = trim(Process::run("which suricata")->output());
        // Build Suricata command
        $cmd = "sudo -u nna $suricataBin -r $this->pcapPath --set outputs.1.eve-log.filename=$outputFilePath -l $logBasePath -c $suricataYamlPath/suricata.yaml";
        $suricataCommand = Process::run($cmd);
        if ($suricataCommand->failed()) {
            $error = $suricataCommand->errorOutput();
            event(new SuricataPcapScanUpdate('failed', $error));
            \Log::error("Suricata failed to process pcap: $error");
            return;
        }
        \Log::info("Suricata Import Pcap Scan Complete");
        // Clean up PCAP file
        if ($this->deletePcap){
            @unlink($this->pcapPath);
        }

        // Update Suricata config with new eve.json file
        $suricata->evelog_name = $outputFileName;
        $suricata->save();

        event(new SuricataPcapScanUpdate('success', "Scan finished, output saved to $outputFileName"));
    }
}