<?php

namespace App\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Process;
use Symfony\Component\Yaml\Yaml;
use Illuminate\Support\Facades\Log;
use App\Models\SuricataRuleset;
use App\Models\SuricataConfig;
use Throwable;

class SyncRulesetsJob implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new job instance.
     */
    public function __construct()
    {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        \Log::info("Running sync job for Suricata rulesets");
        $rulesetNames = SuricataRuleset::pluck('name')->toArray();
        $sourcesDir = str_replace('/rules','/update/sources',rtrim(SuricataConfig::first()->suricata_rules_path, '/'));
        \Log::info($sourcesDir);
        $sources = Process::run("find {$sourcesDir} -type f -name '*.yaml*'");
        if ($sources->failed()){
            \Log::error("Failed to get sources: " . $sources->errorOutput());
        }
        $files = explode("\n", trim($sources->output()));

        foreach ($files as $filePath) {
            \Log::info($filePath);
            if (!file_exists($filePath)) {
                continue;
            }

            try {
                $data = Yaml::parseFile($filePath);
                $secretCodeReq = false;
                $secretCode = '';
                if (!isset($data['source'])) {
                    Log::warning("Skipping file (missing 'source'): $filePath");
                    continue;
                }
                if (isset($data['params']['secret-code'])) {
                    $secretCodeReq = true;
                    $secretCode = $data['params']['secret-code'];
                }
                $source = $data['source'];
                $url = $data['url'] ?? "None";
                $enabled = !str_ends_with($filePath, '.disabled');

                // If the source isn't already in the DB
                if (!in_array($source, $rulesetNames)) {
                    Log::info("Adding new source: {$source}");

                    SuricataRuleset::create([
                        'name' => $source,
                        'url' => $url,
                        'enabled' => $enabled,
                        'description' => "None",
                        'type' => "Custom",
                        'update_frequency' => "None",
                        'secret_code' => $secretCode,
                        'secret_code_required' => $secretCodeReq,
                        'last_updated' => null,
                        'fetch_status' => "None",
                    ]);
                }

            } catch (\Exception $e) {
                Log::error("Failed to parse YAML from $filePath: " . $e->getMessage());
            }
        }
    }
    /**
     * Handle a job failure.
     */
    public function failed(?Throwable $exception)
    {
        Log::error("Suricata sync job failed: " . $exception->getMessage(), [
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'trace' => $exception->getTraceAsString(),
        ]);
    }
}
