<?php

namespace App\Models;

use App\Notifications\Alert;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Config;
use Exception;
use App\Models\SnmpReceivers;
use Illuminate\Notifications\Messages\MailMessage;
use App\Services\NotificationService;

class Check extends Model
{
    public $timestamps = false;

    protected $fillable = [
        'active',
        'name',
        'object_type',
        'object_id',
        'metric',
        'warning',
        'critical',
        'raw_query',
        'last_val',
        'last_run',
        'last_code',
        'last_stdout',
        'check_type',
    ];

    protected $last_code_dict = [
        "OK" => 0,
        "WARNING" => 1,
        "CRITICAL" => 2,
        "UNKNOWN" => 3
    ];

    public function alerting_associations(): HasMany
    {
        return $this->hasMany(AlertingAssociations::class);
    }

    /**
     * get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'object_id' => 'integer',
            'warning' => 'string',
            'critical' => 'string',
        ];
    }

    /**
     * run the check
     * 
     * @return void
     */
    public function run_check($force = false)
    {
        $last_run = date('Y-m-d H:i:s');
        $check_results = [];

        Log::info("Check [ " . $this->name . " (id: " . $this->id . ") ] OF TYPE [ " . $this->check_type . " ] RUNNING CHECK");

        switch ($this->check_type) {
            case "flow_source":
                $check_results = $this->process_flow_data_nfdump($this->object_type, $this->object_id);
                break;
            case "nmap":
                $check_results = $this->process_nmap_scan($this->object_type, $this->object_id);
                break;
            case "suricata":
                $check_results = $this->process_suricata_alerts($force);
                break;
        }

        if (empty($check_results)) {
            return;
        }


        if (!array_key_exists("last_code", $check_results) || !array_key_exists("last_stdout", $check_results) || !array_key_exists("check_object", $check_results)) {
            return;
        }

        // get the alerting associations
        $alerting_associations = $this->get_alerting_associations();

        if (count($alerting_associations['command']) > 0) {
            foreach ($alerting_associations['command'] as $command) {
                $this->run_command($command, $check_results);
            }
        }

        // handle nagios notifications
        if (count($alerting_associations['nagios']) > 0) {
            foreach ($alerting_associations['nagios'] as $nagios) {
                $nagios->send_nrdp_notification($check_results);
            }
        }

        // handle snmp traps
        if (count($alerting_associations['snmp_receiver']) > 0) {
            SnmpReceivers::send_traps($check_results["check_object"]->name, $this->id, $check_results["last_code"], $check_results["last_stdout"]);
        }

        // if the check results are critical or worse, send notifications
        if ($check_results["last_code"] > $this->last_code_dict["OK"]) {
            // handle user notifications
            if (count($alerting_associations['user']) > 0) {
                foreach ($alerting_associations['user'] as $user) {
                    $this->handle_user_notification($user->id, $check_results);
                }
            }
        }


        // update the check row
        $this->update([
            'last_run' => $last_run,
            'last_code' => $check_results["last_code"],
            'last_stdout' => $check_results["last_stdout"]
        ]);

        return $check_results;
    }

    /**
     * process the flow source check
     * 
     * @param Source $source The source
     * @return void
     */
    public function process_flow_data_nfdump($check_object_type, $check_object_id)
    {
        $check_results = [];

        $sources_nfdump_output = [];
        $check_object = null;

        $flow_data_sources = [];
        try {
            // get the check object and the flow data sources
            if ($check_object_type === "source") {
                $check_object = Source::find($check_object_id);
                $flow_data_sources[] = $check_object;
            } else if ($check_object_type === "sourcegroup") {
                $check_object = SourceGroup::find($check_object_id);
                $flow_data_sources = $check_object->sources;
            }

            foreach ($flow_data_sources as $source) {
                // get the last nfcapd file in the source's flows directory
                $source_dir = $source->directory . "/flows/";
                $source_dir_files = preg_grep("*.current.*", scandir($source_dir), PREG_GREP_INVERT);
                $nfcapd_file = $source_dir . $source_dir_files[count($source_dir_files) - 1];

                // run the nfdump query
                $raw_query = $this->raw_query;
                $nfdump_command = "/usr/local/bin/nfdump -r {$nfcapd_file} -o ndjson -s srcip {$raw_query}";

                $nfdump_output = [];
                $nfdump_result_code = 0;
                exec($nfdump_command,  $nfdump_output, $nfdump_result_code);

                if ($nfdump_result_code != 0) {
                    Log::error("Check [ " . $this->name . " (id: " . $this->id . ") ] NFDUMP ERROR -> " . $nfdump_output);
                    return $check_results;
                }

                $sources_nfdump_output = array_merge($sources_nfdump_output, $nfdump_output);
            }

            // get the aggregate data
            $aggregate_data = $this->get_aggregate($sources_nfdump_output, $this->metric);
            if (!$aggregate_data) {
                return $check_results;
            }

            // check the thresholds and generate the respective "last_code"
            $last_code = $this->check_thresholds($aggregate_data, $this->warning, $this->critical);
            if ($last_code === null) {
                return $check_results;
            }

            // generate the "last_stdout"
            $last_stdout = "$this->metric on $check_object->name with filter [$this->raw_query] is $aggregate_data" . " | " . $this->generate_perf_data($aggregate_data, $this->metric);

            $check_results = [
                "check_object" => $check_object,
                "last_code" => $last_code,
                "last_stdout" => $last_stdout
            ];

            return $check_results;
        } catch (Exception $e) {
            Log::error('Check [ ' . $this->name . ' (id: ' . $this->id . ') ] FLOW DATA PROCESSING ERROR -> ' . $e->getMessage());
            return null;
        }
    }

    /**
     * process the nmap scan check
     * 
     * @param string $check_object_type The check object type
     * @param int $check_object_id The check object id
     * @return array The check results
     */
    private function process_nmap_scan($check_object_type, $check_object_id)
    {
        $check_results = [];

        // get the associated scheduled nmap scan
        $check_object = ScheduledNmapScan::find($check_object_id);
        if (!$check_object) {
            Log::error("Check [ " . $this->name . " (id: " . $this->id . ") ] NO SCHEDULED NMAP SCAN FOUND!");
            return $check_results;
        }

        // set default check results values
        $check_results["check_object"] = $check_object;
        $check_results["last_code"] = 3;
        $check_results["last_stdout"] = "unknown";

        // get the latest scan for the scheduled nmap scan
        $latest_scan = $check_object->latestScan();
        if (!$latest_scan) {
            Log::error("Check [ " . $this->name . " (id: " . $this->id . ") ] NO LATEST NMAP SCAN FOUND!");
            return $check_results;
        }

        $nmap_path = "/usr/local/nmap/";
        $nmap_output = $nmap_path . $latest_scan->output_filename . ".json";

        $nmap_output_json = json_decode(file_get_contents($nmap_output), true);

        $nmap_host_array = $nmap_output_json["host"];

        // we need to make sure the hosts are always in an array,
        // even if there is only one host - thanks Nmap :|
        $formatted_nmap_hosts = [];
        if (!array_key_exists("0", $nmap_host_array)) {
            $formatted_nmap_hosts[] = $nmap_host_array;
        } else {
            $formatted_nmap_hosts = $nmap_host_array;
        }

        if (count($nmap_host_array) == 0) {
            Log::error("Check [ " . $this->name . " (id: " . $this->id . ") ] NO NMAP HOST ARRAY FOUND!");
            return $check_results;
        }

        $port_data = $this->get_nmap_port_data($formatted_nmap_hosts);
        switch ($this->metric) {
            case "ports_open":
                $ports_open_count = $this->count_array_values($port_data, "port_state", "open");
                $last_code = $this->check_thresholds($ports_open_count, $this->warning, $this->critical);
                $last_stdout = "Total ports open: $ports_open_count" . " | " . $this->generate_perf_data($ports_open_count, "ports");
                break;
            case "ports_closed":
                $ports_closed_count = $this->count_array_values($port_data, "port_state", "closed");
                $last_code = $this->check_thresholds($ports_closed_count, $this->warning, $this->critical);
                $last_stdout = "Total ports closed: $ports_closed_count" . " | " . $this->generate_perf_data($ports_closed_count, "ports");
                break;
            default:
                $last_code = 3;
                $last_stdout = "Unknown Metric";
                break;
        }

        if ($last_code === null) {
            return $check_results;
        }

        $check_results = [
            "check_object" => $check_object,
            "last_code" => $last_code,
            "last_stdout" => $last_stdout
        ];

        return $check_results;
    }

    /**
     * process the suricata alerts check
     * 
     * @param bool $force Whether to force the check to run
     * @return array The check results
     */
    private function process_suricata_alerts($force = false)
    {
        $check_results = [
            "check_object" => $this->object_id,
            "last_code" => 3,
            "last_stdout" => "No suricata alerts found"
        ];

        $suricata_query_params = $this->parse_suricata_raw_query($this->raw_query);
        $scan_frequency = $suricata_query_params["frequency"];
        $lookback_days = (int) str_replace("d", "", $suricata_query_params["lookback_period"]);

        // if the check is not being forced, check if the check should run
        if (!$force) {
            if (!$this->should_run_check($suricata_query_params)) {
                return $check_results;
            }
        }


        if ($this->object_type === "signature_id") {
            switch ($this->metric) {
                case "alert_count":
                    $suricata_alert = SuricataAlert::where('last_seen', '>=', now('UTC')->subDays($lookback_days))->where('signature_id', $this->object_id)->first();
                    if (!$suricata_alert) {
                        return $check_results;
                    }
                    
                    $last_code = $this->check_thresholds($suricata_alert->last_day_count, $this->warning, $this->critical);
                    if ($last_code === null) {
                        return $check_results;
                    }

                    $last_stdout = "Suricata alerts: " . $suricata_alert->last_day_count;
                    $last_stdout .= " | " . $this->generate_perf_data($suricata_alert->last_day_count, "alerts");
                    break;
                default:
                    $last_code = 3;
                    $last_stdout = "Unknown Metric";
                    break;
            }
        }


        $check_results = [
            "check_object" => $this->object_id,
            "last_code" => $last_code,
            "last_stdout" => $last_stdout
        ];

        return $check_results;
    }


    /**
     * parse the suricata raw query
     * 
     * @param string $raw_query The raw query
     * @return array The suricata query params
     */
    private function parse_suricata_raw_query($raw_query)
    {
        $suricata_raw_query = explode("&", $raw_query);

        $suricata_query_params = [];
        foreach ($suricata_raw_query as $query) {
            $query = explode("=", $query);
            $suricata_query_params[$query[0]] = $query[1];
        }
        return $suricata_query_params;
    }

    /**
     * check if the check should run
     * 
     * @param array $suricata_query_params The suricata query params
     * @return bool True if the check should run, false otherwise
     */
    private function should_run_check($suricata_query_params)
    {
        // get the frequency value from the query params
        if (!isset($suricata_query_params['frequency'])) {
            return true;
        }

        $frequency = $suricata_query_params['frequency'];

        // parse the frequency string (e.g., "5min", "1h", "30s")
        $pattern = '/^(\d+)(s|min|h|d|w|m|y)$/';
        if (!preg_match($pattern, $frequency, $matches)) {
            return true;
        }

        $interval_value = (int)$matches[1];
        $interval_unit = $matches[2];

        // convet frequency to seconds
        switch ($interval_unit) {
            case 's':
                $interval_seconds = $interval_value;
                break;
            case 'min':
                $interval_seconds = $interval_value * 60;
                break;
            case 'h':
                $interval_seconds = $interval_value * 3600;
                break;
            case 'd':
                $interval_seconds = $interval_value * 86400;
                break;
            case 'w':
                $interval_seconds = $interval_value * 604800;
                break;
            case 'm':
                $interval_seconds = $interval_value * 2592000;
                break;
            case 'y':
                $interval_seconds = $interval_value * 31536000;
                break;
            default:
                return true;
        }

        // no last run means the check hasn't run yet
        if (empty($this->last_run)) {
            return true;
        }

        // get the last run timestamp as unix time
        $last_run_timestamp = strtotime($this->last_run);

        if ($last_run_timestamp === false) {
            return true;
        }

        $current_time = now('UTC')->timestamp;

        $elapsed = $current_time - $last_run_timestamp;

        // return whether enough time has passed
        return $elapsed >= $interval_seconds;
    }

    /**
     * get the port data from an nmap scan
     * 
     * @param array $host_array The host array
     * @return array The port data
     */
    private function get_nmap_port_data($host_array)
    {
        $port_data = [];
        foreach ($host_array as $host) {
            if (!array_key_exists("ports", $host)) {
                continue;
            }
            $port_array = $host["ports"]["port"];
            if (count($port_array) == 0) {
                continue;
            }
            foreach ($port_array as $port) {
                $port_state = $port["state"]["state"];
                $port_id = $port["portid"];
                $port_protocol = $port["protocol"];
                $port_service = $port["service"]["name"];
                $port_service_version = "N/A";
                if (array_key_exists("version", $port["service"])) {
                    $port_service_version = $port["service"]["version"];
                }

                $port_data[] = [
                    "port_state" => $port_state,
                    "port_id" => $port_id,
                    "port_protocol" => $port_protocol,
                    "port_service" => $port_service,
                    "port_service_version" => $port_service_version
                ];
            }
        }
        return $port_data;
    }

    /**
     * count the number of values in an array
     * 
     * @param array $array The array to count the values of
     * @param string $key The key to count the values of
     * @param string $value The value to count
     * @return int The number of values in the array
     */
    private function count_array_values($array, $key, $value)
    {
        $count = 0;
        foreach ($array as $item) {
            if ($item[$key] === $value) {
                $count++;
            }
        }
        return $count;
    }

    /**
     * send the command to the command
     * 
     * @param array $check_results The check results
     * @return void
     */
    private function run_command($command, $check_results)
    {
        $command_args = "";
        $command_output = "";
        try {
            $command_script = $command->script;
            $command_arguments = $command->arguments;
            $command_location = $command->location;
            $command_name = $command->name;
            $command_id = $command->id;


            // parse pre-defined command arg macros
            if ($this->object_type === "source") {
                $command_args = str_replace("%sourcename%", '"' . $check_results["check_object"]->name . '"', $command_arguments);
            }

            if ($this->object_type === "sourcegroup") {
                $command_args = str_replace("%sourcegroupname%", '"' . $check_results["check_object"]->name . '"', $command_arguments);
            }

            if ($this->check_type === "suricata") {
                $command_args = str_replace("%alert_count%", '"' . $check_results["check_object"]->last_day_count . '"', $command_args);
            }

            $command_args = str_replace("%state%", '"' . array_search($check_results["last_code"], $this->last_code_dict) . '"', $command_args);
            $command_args = str_replace("%returncode%", '"' . $check_results["last_code"] . '"', $command_args);
            $command_args = str_replace("%output%", '"' . $check_results["last_stdout"] . '"', $command_args);

            $command_output = shell_exec($command_location . "/" . $command_script . " " . $command_args);
            Log::info("Check [ " . $this->name . " (id: " . $this->id . ") ] COMMAND [ " . $command_name . " (id: " . $command_id . ") ] OUTPUT -> " . $command_output);
            return;
        } catch (Exception $e) {
            Log::error('Check [ ' . $this->name . ' (id: ' . $this->id . ') ] COMMAND EXECUTION ERROR -> ' . $e->getMessage());
            return;
        }
    }

    /**
     * handle user notifications
     * 
     * @param int $user_id The user id
     * @param array $check_results The check results
     * @return void
     */
    private function handle_user_notification($user_id, $check_results)
    {
        try {
            $user = User::find($user_id);
            $notification_options = NotificationOptions::first();

            if (!$notification_options) {
                Log::error("Check [ " . $this->name . " (id: " . $this->id . ") ] NO NOTIFICATION OPTIONS FOUND!");
                return;
            }

            $notification_options->alert_title = "Check [$this->name] Alert";
            $notification_options->alert_message = "Check [$this->name] Results: $check_results[last_stdout]";
            $notification_options->alert_type = "check";

            // disable broadcasting on check alerts for now
            $notification_options->notification_channel = ['mail'];

            NotificationService::configure($notification_options, $user);

            $alert_child_page_location = "/alerting";
            if ($this->check_type === "nmap") {
                $alert_child_page_location = "/alerting#nmap";
            }
            if ($this->check_type === "suricata") {
                $alert_child_page_location = "/alerting#suricata";
            }

            $notification_options->mail_message = (new MailMessage)
                ->mailer($notification_options->mail_config['mailer'])
                ->subject("Alert: Check [{$this->name}] is " . strtoupper(array_search($check_results['last_code'], $this->last_code_dict)))
                ->greeting('Attention: Check Alert')
                ->line('The following check has returned a WARNING or CRITICAL status.')
                ->line('Check Name: ' . $this->name)
                ->line('Status: ' . strtoupper(array_search($check_results['last_code'], $this->last_code_dict)))
                ->line('Timestamp: ' . now('UTC')->toDateTimeString())
                ->line("Results:\n" . $check_results['last_stdout'])
                ->action('View Alerts', url($alert_child_page_location))
                ->line('Please review the check and take any necessary action.');

            $user->notify(new Alert($notification_options));
        } catch (Exception $e) {
            Log::error('Check [ ' . $this->name . ' (id: ' . $this->id . ') ] USER NOTIFICATION ERROR -> ' . $e->getMessage());
        }
    }

    /**
     * get the alerting associations for a check
     * 
     * @return array The alerting associations
     */
    private function get_alerting_associations()
    {
        $alerting_associations = AlertingAssociations::where('check_id', $this->id)->get();

        // group alerting associations by association_type
        $grouped_associations = [
            'user' => [],
            'snmp_receiver' => [],
            'command' => [],
            'nagios' => [],
        ];

        foreach ($alerting_associations as $association) {
            $type = $association->association_type;
            $association_id = $association->association_id;

            switch ($type) {
                case 'user':
                    $grouped_associations['user'][] = User::find($association_id);
                    break;
                case 'snmp_receiver':
                    $grouped_associations['snmp_receiver'][] = SnmpReceivers::find($association_id);
                    break;
                case 'command':
                    $grouped_associations['command'][] = Commands::find($association_id);
                    break;
                case 'nagios':
                    $grouped_associations['nagios'][] = ServiceHostnames::find($association_id);
                    break;
                default:
                    Log::error("Unknown association type: " . $type);
                    break;
            }
        }

        return $grouped_associations;
    }

    /**
     * generate the performance data
     * 
     * @param float $aggregate_data The aggregate data
     * @return string The performance data
     */
    private function generate_perf_data($aggregate_data, $unit_of_measurement = "")
    {
        try {
            $label = $this->metric;
            $value = $aggregate_data;

            // ensure value is numeric
            if (!is_numeric($value)) {
                $value = 0;
            }

            $uom = $unit_of_measurement;
            $warn = $this->warning ?? "";
            $crit = $this->critical ?? "";
            $min = "";
            $max = "";

            $perfdata = sprintf(
                "'%s'=%s%s;%s;%s;%s;%s",
                $label,
                $value,
                $uom,
                $warn,
                $crit,
                $min,
                $max
            );

            // remove trailing semi-colons
            $perfdata = rtrim($perfdata, ';');

            return $perfdata;
        } catch (Exception $e) {
            Log::error('Check [ ' . $this->name . ' (id: ' . $this->id . ') ] PERFORMANCE DATA GENERATION ERROR -> ' . $e->getMessage());
            return null;
        }
    }

    /**
     * get the aggregate data
     * 
     * @param array $nfdump_output The nfdump output
     * @param string $metric The metric
     * @return float The aggregate data
     */
    private function get_aggregate($nfdump_output, $metric)
    {
        try {
            $metric_aggregate = 0;
            foreach ($nfdump_output as $output) {
                $output = json_decode(trim($output), true);

                // if a sourgegroup is used, there could be a "No Matching Flows" string...
                // if this is the case, we don't want to include it in the aggregate
                if (!is_array($output)) {
                    continue;
                }
                // if the metric doesn't exist, we don't want to include it in the aggregate
                if (!array_key_exists($metric, $output)) {
                    continue;
                }

                $metric_aggregate += $output[$metric];
            }

            return $metric_aggregate;
        } catch (Exception $e) {
            Log::error('Check [ ' . $this->name . ' (id: ' . $this->id . ') ] DATA AGGREGATION ERROR -> ' . $e->getMessage());
            return null;
        }
    }

    /**
     * check the thresholds
     * 
     * @param float $rawval The raw value
     * @param float $warning The warning threshold
     * @param float $critical The critical threshold
     * @return int The last code
     */
    private function check_thresholds($rawval, $warning, $critical)
    {
        try {
            $critical_result = $this->is_within_range($critical, $rawval);
            $warning_result = $this->is_within_range($warning, $rawval);

            if ($critical_result) {
                $last_code = 2;
            } else if ($warning_result) {
                $last_code = 1;
            } else {
                $last_code = 0;
            }

            return $last_code;
        } catch (Exception $e) {
            Log::error('Check [ ' . $this->name . ' (id: ' . $this->id . ') ] THRESHOLD CHECK ERROR -> ' . $e->getMessage());
            return null;
        }
    }

    /**
     * check if the value is within the range
     * 
     * @param string $nagstring The nagstring
     * @param float $value The value
     * @return bool True if the value is within the range, false otherwise
     */
    private function is_within_range($nagstring, $value)
    {
        try {
            if (empty($nagstring)) {
                return false;
            }

            // trim the nagstring incase of whitespace
            $nagstring = trim($nagstring);

            $value = (float) $value;

            // regex patterns and their corresponding logic
            $actions = [
                // single value: ^number$ -> value > threshold OR value < 0
                ['/^(-?[0-9]+(\.[0-9]+)?)$/', function ($matches) use ($value) {
                    return ($value > (float) $matches[1]) || ($value < 0);
                }],

                // upper bound: ^number:$ -> value < threshold
                ['/^(-?[0-9]+(\.[0-9]+)?):$/', function ($matches) use ($value) {
                    return $value < (float) $matches[1];
                }],

                // lower bound: ^~:number$ -> value > threshold
                ['/^~:(-?[0-9]+(\.[0-9]+)?)$/', function ($matches) use ($value) {
                    return $value > (float) $matches[1];
                }],

                // range: ^number:number$ -> value < min OR value > max
                ['/^(-?[0-9]+(\.[0-9]+)?):(-?[0-9]+(\.[0-9]+)?)$/', function ($matches) use ($value) {
                    return ($value < (float) $matches[1]) || ($value > (float) $matches[3]);
                }],

                // inverted range: ^@number:number$ -> NOT (value < min OR value > max)
                ['/^@(-?[0-9]+(\.[0-9]+)?):(-?[0-9]+(\.[0-9]+)?)$/', function ($matches) use ($value) {
                    return !(($value < (float) $matches[1]) || ($value > (float) $matches[3]));
                }]
            ];

            foreach ($actions as [$pattern, $callback]) {
                if (preg_match($pattern, $nagstring, $matches)) {
                    return $callback($matches);
                }
            }

            // if no pattern matches, return false (meaning the value is within acceptable range)
            return false;
        } catch (Exception $e) {
            // handle the exception 
            Log::error('is_within_range operation failed: ' . $e->getMessage());
        }
    }

    // ABNORMAL BEHAVIOR FUNCTIONS (for when this is set-up)
    /**
     * get failures metric for a given source id
     * 
     * @param int $sid source id
     * @return float|null failure metric value or null if no data
     */
    private function get_failures($source)
    {
        // get the rrd file path for this source
        $bandwidth_path = $source->directory . "/bandwidth.rrd";

        // define the metrics we want to query
        $columns = ['bytes', 'flows', 'packets'];

        // build rrdtool command arguments
        $rrd_args = ['--json', '--start', '-600', '--end', '-300'];

        // add definitions and exports for each metric
        foreach ($columns as $index => $col) {
            $rrd_args[] = $this->make_defs($col, $bandwidth_path, '', 'FAILURES');
            $rrd_args[] = $this->make_xport($col, $index);
        }

        // execute rrdtool command
        $command = 'rrdtool xport ' . implode(' ', array_map('escapeshellarg', $rrd_args));

        Log::debug("RRD COMMAND -> " . $command);

        $output = shell_exec($command);

        // parse json result
        $result = json_decode($output, true);

        // if we don't get data back there was something wrong with our query
        if (!isset($result['data'])) {
            return null;
        }

        // set result to a list with one entry
        $result = $result['data'];

        // get the only entry (a 3-tuple)
        $result = $result[0];

        // reduce this to just 1 float by summing all values
        $result = (float) array_sum($result);

        return $result;
    }

    /**
     * make rrd definition string
     * 
     * @param string $metric metric name
     * @param string $rrd rrd file path
     * @param string $metric_prefix prefix for metric
     * @param string $round_robin_archive rra type
     * @return string rrd definition string
     */
    private function make_defs($metric, $rrd, $metric_prefix = '', $round_robin_archive = 'AVERAGE')
    {
        $parts = [
            'DEF',
            $metric_prefix . $metric . '=' . $rrd,
            $metric,
            $round_robin_archive
        ];
        return implode(':', $parts);
    }

    /**
     * make rrd export string
     * 
     * @param string $key export key
     * @param int $label export label
     * @return string rrd export string
     */
    private function make_xport($key, $label)
    {
        $parts = [
            'XPORT',
            $key,
            "'" . $label . "'"
        ];
        return implode(':', $parts);
    }
}
