<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Models\NagiosServers;
use DOMDocument;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Exception;

class ServiceHostnames extends Model
{
    public $timestamps = false;

    protected $fillable = [
        'hostname',
        'servicename',
        'server_id',
    ];

    public function nagios_server(): BelongsTo
    {
        return $this->belongsTo(NagiosServers::class, 'server_id');
    }

    /**
     * get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'server_id' => 'integer',
        ];
    }

    /**
     * creates NRDP XML for sending check results to Nagios servers
     * 
     * @param string $hostname The hostname for the check
     * @param string $servicename The service name for the check
     * @param int $returncode The return code (0=OK, 1=WARNING, 2=CRITICAL, 3=UNKNOWN)
     * @param string $stdout The check output message
     * @return string The XML string for NRDP submission
     */
    public function make_nrdp_xml($hostname, $servicename, $returncode, $stdout)
    {
        // create the xml document
        $xml = new DOMDocument('1.0', 'UTF-8');
        $xml->formatOutput = true;

        // create root element
        $checkresults = $xml->createElement('checkresults');
        $xml->appendChild($checkresults);

        // create checkresult element with type attribute
        $checkresult = $xml->createElement('checkresult');
        $checkresult->setAttribute('type', 'service');
        $checkresults->appendChild($checkresult);

        // create hostname element
        $xhostname = $xml->createElement('hostname');
        $xhostname->textContent = $hostname;
        $checkresult->appendChild($xhostname);

        // create servicename element
        $xservicename = $xml->createElement('servicename');
        $xservicename->textContent = $servicename;
        $checkresult->appendChild($xservicename);

        // create state element
        $state = $xml->createElement('state');
        $state->textContent = (string)$returncode;
        $checkresult->appendChild($state);

        // create output element
        $output = $xml->createElement('output');
        $output->textContent = $stdout;
        $checkresult->appendChild($output);

        // return the xml as string
        return $xml->saveXML();
    }


    /**
     * send an NRDP notification to the Nagios server
     * 
     * @param array $check_results The check results
     * @return string The response from the Nagios server
     */
    public function send_nrdp_notification($check_results)
    {
        try {
            $url = $this->nagios_server->nrdp_url;
            $token = $this->nagios_server->nrdp_token;
            $xml_data = $this->make_nrdp_xml($this->hostname, $this->servicename, $check_results['last_code'], $check_results['last_stdout']);

            $post_data = [
                'XMLDATA' => $xml_data,
                'token' => $token,
                'cmd' => 'submitcheck',
            ];

            // need to be explicit or NRDP won't process the check results correctly
            $response = Http::asForm()
                ->withHeaders([
                    'Accept' => 'application/xml, text/xml',
                    'User-Agent' => 'Laravel-NRDP-Client/1.0'
                ])
                ->post($url, $post_data);

            return $response->body();
        } catch (Exception $e) {
            Log::error('ServiceHostnames [ ' . $this->hostname . ' (id: ' . $this->id . ') ] NRDP NOTIFICATION ERROR -> ' . $e->getMessage());
            return null;
        }
    }
}
