<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;
use Exception;
use App\Models\AlertingAssociations;

class SnmpReceivers extends Model
{
    public $timestamps = false;

    protected $fillable = [
        'name',
        'version',
        'community_string',
        'auth_level',
        'priv_protocol',
        'priv_password',
        'auth_protocol',
        'auth_password',
        'username',
        'port',
        'ip',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'version' => 'string',
            'port' => 'integer',
        ];
    }

    /**
     * send snmp traps for a given check
     * 
     * @param string $source_name name of the source
     * @param int $check_id check id
     * @param int $return_code return code (0=OK, 1=WARNING, 2=CRITICAL, 3=UNKNOWN)
     * @param string $stdout check output message
     * @return void
     */
    public static function send_traps($source_name, $check_id, $return_code, $stdout)
    {
        try {
            // query to get trap receiver ids linked to this check
            $alerting_associations = AlertingAssociations::where('check_id', $check_id)
                ->where('association_type', 'snmp_receiver')
                ->get();

            foreach ($alerting_associations as $association) {
                $snmp_receiver = self::find($association->association_id);

                // check if we got valid data
                if (!$snmp_receiver) {
                    Log::error("SnmpReceivers [ " . $source_name . " (id: " . $check_id . ") ] NO SUCH SNMP RECEIVER -> " . $association->association_id);
                    continue;
                }

                // build the snmp trap command
                $snmp_cmd = self::build_snmp_trap_cmd($snmp_receiver, $source_name, $return_code, $stdout);
                exec($snmp_cmd, $output, $return_var);

                // check if the snmp trap command failed
                if ($return_var !== 0) {
                    Log::error("SnmpReceivers [ " . $source_name . " (id: " . $check_id . ") ] SNMP TRAP ERROR -> " . $output);
                }
            }
        } catch (Exception $e) {
            Log::error('SnmpReceivers [ ' . $source_name . ' (id: ' . $check_id . ') ] SNMP TRAP ERROR -> ' . $e->getMessage());
        }
    }

    /**
     * build snmp trap command
     * 
     * @param SnmpReceivers $snmp_receiver trap receiver configuration from database
     * @param string $source_name name of the source
     * @param int $return_code return code
     * @param string $stdout check output message
     * @return string complete snmptrap command
     */
    private static function build_snmp_trap_cmd($snmp_receiver, $source_name, $return_code, $stdout)
    {
        // start building the command
        $cmd_parts = ['/usr/bin/snmptrap'];

        // add version-specific parameters
        if ($snmp_receiver->version == '2c') {
            $cmd_parts[] = '-v2c';
            $cmd_parts[] = '-c';
            $cmd_parts[] = escapeshellarg($snmp_receiver->community_string);
        } else {
            // snmpv3
            $cmd_parts[] = '-v3';
            $cmd_parts[] = '-u';
            $cmd_parts[] = escapeshellarg($snmp_receiver->username);

            if ($snmp_receiver->auth_level == 'AuthPriv') {
                $cmd_parts[] = '-a';
                $cmd_parts[] = escapeshellarg($snmp_receiver->auth_protocol);
                $cmd_parts[] = '-A';
                $cmd_parts[] = escapeshellarg($snmp_receiver->auth_password);
                $cmd_parts[] = '-x';
                $cmd_parts[] = escapeshellarg($snmp_receiver->priv_protocol);
                $cmd_parts[] = '-X';
                $cmd_parts[] = escapeshellarg($snmp_receiver->priv_password);
            } elseif ($snmp_receiver->auth_level == 'AuthNoPriv') {
                $cmd_parts[] = '-a';
                $cmd_parts[] = escapeshellarg($snmp_receiver->auth_protocol);
                $cmd_parts[] = '-A';
                $cmd_parts[] = escapeshellarg($snmp_receiver->auth_password);
            }
        }

        // add target ip:port
        $cmd_parts[] = escapeshellarg($snmp_receiver->ip . ':' . $snmp_receiver->port);

        // add empty string (enterprise oid placeholder)
        $cmd_parts[] = "''";

        // add nagios service event trap using mib name
        $cmd_parts[] = 'NAGIOS-NOTIFY-MIB::nSvcEvent';

        // add nagios service event variables using mib names
        // nSvcHostname
        $cmd_parts[] = 'nSvcHostname';
        $cmd_parts[] = 's';
        $cmd_parts[] = escapeshellarg('NagiosNA');

        // nSvcHostStateID (host state - use 0 for UP)
        $cmd_parts[] = 'nSvcHostStateID';
        $cmd_parts[] = 'i';
        $cmd_parts[] = 0;

        // nSvcDesc (service description)
        $cmd_parts[] = 'nSvcDesc';
        $cmd_parts[] = 's';
        $cmd_parts[] = escapeshellarg($source_name);

        // nSvcStateID (service state)
        $cmd_parts[] = 'nSvcStateID';
        $cmd_parts[] = 'i';
        $cmd_parts[] = $return_code;

        // nSvcOutput (check output)
        $cmd_parts[] = 'nSvcOutput';
        $cmd_parts[] = 's';
        $cmd_parts[] = escapeshellarg($stdout);

        // join all parts into a single command string
        return implode(' ', $cmd_parts);
    }
}
