<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Process;
use App\Models\ScheduledNmapScan;
use App\Models\Report;
use App\Models\NmapScan;
use App\Models\ConfigOption;
use App\Models\SuricataConfig;
use App\Models\SuricataAlert;
use App\Models\SuricataScan;
use App\Jobs\NmapScanJob;
use App\Jobs\ReportJob;
use App\Jobs\CheckJob;
use App\Events\NmapScanCreated;
use App\Models\Check;
use Cron\CronExpression;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\Eloquent\Relations\Relation;
use App\Models\User;
use App\Models\NagiosServers;
use App\Models\SnmpReceivers;
use App\Models\Commands;
use App\Services\ScheduleService;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Bus;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap any application services.
     * 
     * THIS FUNCTION MUST RETURN EARLY IF ANY DATABASE TABLE USED WITHIN DOES NOT EXIST!!! 
     * Laravel calls this function before migrations are ran! 
     */
    public function boot(): void
    {
        if (!Schema::hasTable('nagiosna.cache') ||
            !Schema::hasTable('scheduled_nmap_scans') ||
            !Schema::hasTable('suricata_rulesets') ||
            !Schema::hasTable('nagiosna.suricata_scan') || 
            !Schema::hasTable('reports'))
        {
            return;
        }
        // // if (app()->runningInConsole()) return;
        
        $timezone = Cache::remember('config.timezone', 60, function () {
            return ConfigOption::where('name', 'timezone')->value('value') ?? 'UTC';
        });

        Config::set('app.timezone', $timezone);
        Config::set('app.schedule_timezone', $timezone);
        date_default_timezone_set($timezone);

        Relation::morphMap([
            'user' => User::class,
            'nagios' => NagiosServers::class,
            'snmp_receiver' => SnmpReceivers::class,
            'command' => Commands::class,
        ]);

        // Define Gates for API routes/controllers that do not utilize a policy
        Gate::define('accessBackupController', function (User $user) {
            return $user->is_admin;
        });

        Gate::define('accessSystemController', function (User $user) {
            return true;
        });

        Gate::define('createQuery', function (User $user) {
            return true;
        });

        $this->app->booted(function () {
            $schedule = app(Schedule::class);
            $timezone = ConfigOption::where('name', 'timezone')->value('value');

            $suricataConfig = SuricataConfig::first();
            if ($suricataConfig && $suricataConfig->rulesets_seeded) {
                $schedule->command('rulesets:sync')->hourly()->name('Sync Rulesets')->timezone($timezone); // sync the sources file to the DB every hour
                if ($suricataConfig->auto_update_rules){
                    $schedule->command('rulesets:update 5min')->everyFiveMinutes()->name('Update Ruleset: 5min')->timezone($timezone);
                    $schedule->command('rulesets:update Hourly')->hourly()->name('Update Ruleset: Hourly')->timezone($timezone); // runs every hour
                    $schedule->command('rulesets:update Daily')->dailyAt('00:00')->name('Update Ruleset: Daily')->timezone($timezone); //runs on midnight everyday
                    $schedule->command('rulesets:update Weekly')->weeklyOn(1, '00:00')->name('Update Ruleset: Weekly')->timezone($timezone); //runs on monday 12am
                    $schedule->command('rulesets:update Monthly')->monthlyOn(1, '00:00')->name('Update Ruleset: Monthly')->timezone($timezone); //runs at 12am 1st of the month
                }
            }

            $suricataScan = SuricataScan::firstOrCreate(
                [],
                [
                    'pid' => null,
                    'running' => false,
                    'last_rotated' => null,
                    'started_at' => null,
                    'eve_current_line' => 0,
                    'current_eve_log' => null,
                ]
            );

            if ($suricataScan->isRunning()) {
                $schedule->command('alerts:suricata')->everyMinute()->name('Get suricata Alerts')->timezone($timezone);
            } 

            if (SuricataAlert::exists()){ 
                $schedule->command('alerts:suricata-rollover')->hourly()->name('Rotate alerts')->timezone($timezone);
            }

            $scheduledScans = ScheduledNmapScan::where('enabled', true)->get();
            foreach ($scheduledScans as $scheduledScan) {
                $type = $scheduledScan->schedule_type;
                $params = $scheduledScan->schedule_parameters ?? [];

                // Use fields directly instead of building 'args'
                if ($type === 'cron' && !empty($params['expression'])) {
                    try {
                        CronExpression::factory($params['expression']);
                    } catch (\InvalidArgumentException $e) {
                        Log::warning("Invalid cron for scan ID {$scheduledScan->id}: {$e->getMessage()}");
                        continue;
                    }
                }

                $event = $schedule->call(function () use ($scheduledScan) {
                    $scan = NmapScan::create([
                        'user_id' => $scheduledScan->user_id,
                        'nmap_profile_id' => $scheduledScan->nmap_profile_id,
                        'scheduled_nmap_scan_id' => $scheduledScan->id,
                        'title' => $scheduledScan->name . ' [' . now()->format('Y-m-d H:i:s' . ']'),
                        'status' => 'Pending',
                        'scan_parameters' => $scheduledScan->parameters,
                    ]);

                    $associated_checks = Check::where('active', true)
                        ->where('object_type', 'scheduled_scan')
                        ->where('object_id', $scheduledScan->id)
                        ->get();

                    Bus::chain([
                        new NmapScanJob($scan),
                        ...$associated_checks->map(function ($check) {
                            return new CheckJob($check);
                        }),
                    ])->dispatch();
                    NmapScanCreated::dispatch($scan);

                    Log::info("Scheduled scan run", [
                        'scheduled_id' => $scheduledScan->id,
                        'scan_id' => $scan->id,
                    ]);
                })->name("Scheduled Nmap: {$scheduledScan->name}");

                if (!ScheduleService::applyTypeToEvent($type, $event, $params, $timezone)) {
                    continue;
                }
            }

            $reports = Report::where('enabled', true)->get();
            foreach ($reports as $report) {
                $type = $report->schedule;
                $params = $report->schedule_parameters ?? [];

                // Use fields directly instead of building 'args'
                if ($type === 'cron' && !empty($params['expression'])) {
                    try {
                        CronExpression::factory($params['expression']);
                    } catch (\InvalidArgumentException $e) {
                        Log::warning("Invalid cron for scan ID {$report->id}: {$e->getMessage()}");
                        continue;
                    }
                }

                $event = $schedule->call(function () use ($report) {
                    ReportJob::dispatch($report);
                })->name("Report: {$report->name}");

                if (!ScheduleService::applyTypeToEvent($type, $event, $params, $timezone)) {
                    continue;
                }
            }

            if (Schema::hasTable('checks')) {
                $active_checks = Check::where('active', true)->get();
                foreach ($active_checks as $check) {
                    $event = $schedule->call(function () use ($check) {
                        CheckJob::dispatch($check);

                        Log::debug("Scheduled check job", [
                            'check_id' => $check->id,
                            'check_name' => $check->name,
                        ]);
                    })->name("Scheduled Check: {$check->name}")->everyFiveMinutes();
                }
            }
        });
    }
}
