import { Routes, Route, useSearchParams } from 'react-router'
import './App.css'
import routes from './routes'
import Layout from './components/Layout'
import MinimalLayout from "@/components/MinimalLayout";
import RouteProtector from "@/components/RouteProtector";
import { configureEcho } from "@laravel/echo-react";
import axios from 'axios'
import { useEnvironment } from "@/contexts/EnvironmentContext";
import { useEffect, useState } from 'react';

function App() {

  const [searchParams] = useSearchParams();
  const { environment } = useEnvironment();
  const [echoReady, setEchoReady] = useState(false);

  useEffect(() => {
    if (environment) {
      configureEcho({
        broadcaster: "reverb",
        key: environment.reverbAppKey,
        wsHost: window.location.hostname,
        wsPort: 8080,
        wssPort: 8080,
        forceTLS: false,
        enabledTransports: ['ws', 'wss'],
        authorizer: (channel) => {
          return {
            authorize: (socketId, callback) => {
              axios.post('/api/broadcasting/auth', {
                socket_id: socketId,
                channel_name: channel.name
              })
              .then(response => callback(null, response.data))
              .catch(error => callback(new Error('Failed to get environment.'), error));
            }
          };
        },
      });
      setEchoReady(true);
    }
  }, [environment]);
  
  if (!echoReady) return <></>;

  const LayoutComponent = searchParams.get('export') === 'true' ? MinimalLayout : Layout;

  return (
    <>
      <Routes>
        {routes.map((route) => {
          if (route.path === "/login") {
            return (
              <Route element={<RouteProtector type="login" />}>
                <Route
                  key={route.path}
                  path={route.path}
                  element={<MinimalLayout>{route.element}</MinimalLayout>}
                />
              </Route>
            )
          } else if (route.path === "/install") {
            return (
              <Route element={<RouteProtector type="install" />}>
                <Route
                  key={route.path}
                  path={route.path}
                  element={<MinimalLayout>{route.element}</MinimalLayout>}
                />
              </Route>
            )
          } else if (route.adminOnly) {
            return (
              <Route element={<RouteProtector type="admin" />} key={route.path}>
                <Route
                  key={route.path}
                  path={route.path}
                  element={<Layout>{route.element}</Layout>}
                />
              </Route>
            )
          } else if (route.path === "/enterkey") {
            return (
              <Route element={<RouteProtector type="enterkey" />}>
                <Route
                  key={route.path}
                  path={route.path}
                  element={<MinimalLayout>{route.element}</MinimalLayout>}
                />
              </Route>
            )
          }
          return (
            <Route element={<RouteProtector type="protected" />}>
              <Route
                key={route.path}
                path={route.path}
                element={<LayoutComponent>{route.element}</LayoutComponent>}
              />
            </Route>
          )
        })}
      </Routes>
    </>
  )
}

export default App
