import { Home, BarChart, FileText, HelpCircle, Layers, Shield, Network, ChevronDown, GlobeLockIcon, Radar, ChevronsUpDown } from "lucide-react"
import { Link, useLocation, useNavigate } from 'react-router';
import { useTranslation } from "react-i18next";
import NagiosIcon from '@/assets/images/nagios-n.svg';
import { useAuth } from "@/contexts/AuthContext";

import {
  Sidebar,
  SidebarContent,
  SidebarGroup,
  SidebarGroupContent,
  SidebarHeader,
  SidebarMenu,
  SidebarMenuButton,
  SidebarMenuItem,
  SidebarFooter,
  SidebarMenuSub,
  SidebarMenuSubItem,
  SidebarMenuSubButton,
} from "@/components/ui/sidebar"

import {
  Collapsible,
  CollapsibleTrigger,
  CollapsibleContent,
} from "@/components/ui/collapsible";

import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"
import axios from "axios";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import SharkFin from "./ui/icons/SharkFin";
import { useGetProductInfo } from "@/components/queries";

// Menu items based on your routes
const items = [
  {
    title: "Flow Sources",
    path: "/sources",
    icon: Network,
  },
  {
    title: "Groups",
    path: "/groups",
    icon: Layers,
  },
  {
    title: "Reports",
    path: "/reports",
    icon: FileText,
  },
  {
    title: "Alerting",
    path: "/alerting",
    icon: BarChart,
  },
  {
    title: "Wireshark",
    path: "/wireshark",
    icon: SharkFin,
  },
  {
    title: "Nmap",
    path: "/nmap",
    icon: Radar,
  },
  {
    title: "Suricata",
    path: "/suricata",
    icon: GlobeLockIcon,
  },
  {
    title: "Help",
    path: "/help",
    icon: HelpCircle,
  },
  // {
  //   title: "Administration",
  //   path: "/administration",
  //   icon: Shield,
  // },
]

const adminItems = [
  {
    title: "Global Settings",
    path: "/admin/globals",
  },
  {
    title: "Notification Settings",
    path: "/admin/notifications",
  },
  {
    title: "License Settings",
    path: "/admin/license",
  },
  {
    title: "User Management",
    path: "/admin/users",
  },
  {
    title: "LDAP/AD Integration",
    path: "/admin/auth_servers",
  },
  {
    title: "System Backup",
    path: "/admin/backup",
  },
]

export function AppSidebar() {
  const location = useLocation();
  const { t } = useTranslation();
  const navigate = useNavigate();
  const { user } = useAuth();
  const queryClient = useQueryClient();
  const { data: productInfo } = useGetProductInfo();

  const logoutMutation = useMutation({
    mutationFn: () => {
      return axios.post('/api/v1/logout');
    },
    onSuccess: async () => {
      await queryClient.invalidateQueries({ queryKey: ['auth'] });
      localStorage.removeItem('login_locale');
      localStorage.removeItem('locale');
      navigate('/login');
    },
  })

  return (
    <Sidebar collapsible="icon">
      <SidebarHeader>
        <SidebarMenu>
          <SidebarMenuItem>
            <SidebarMenuButton
              asChild
              className="py-5 mt-0.5"
            >
              <Link to="/">
                <img src={NagiosIcon} alt="Nagios NA" className="!size-6" />
                <span className="text-lg font-semibold">Network Analyzer</span>
              </Link>
            </SidebarMenuButton>
          </SidebarMenuItem>
        </SidebarMenu>
      </SidebarHeader>
      <SidebarContent>
        <SidebarGroup>
          <SidebarGroupContent>
            <SidebarMenu>
              {user &&
                <SidebarMenuItem key={'dashboards'}>
                  <SidebarMenuButton asChild isActive={location.pathname.slice(0, 11) === '/dashboards'}>
                    <Link to={'/dashboards/' + user.default_dashboard}>
                      <Home />
                      <span>{t('Dashboards')}</span>
                    </Link>
                  </SidebarMenuButton>
                </SidebarMenuItem>
              }
              {items.map((item) => (
                <SidebarMenuItem key={item.title}>
                  <SidebarMenuButton asChild isActive={location.pathname === item.path}>
                    <Link to={item.path}>
                      <item.icon />
                      <span>{t(item.title)}</span>
                    </Link>
                  </SidebarMenuButton>
                </SidebarMenuItem>
              ))}
              {!!user?.is_admin && (
                <Collapsible className="group/collapsible">
                  <SidebarMenuItem>
                    <CollapsibleTrigger asChild>
                      <SidebarMenuButton>
                        <Shield />
                        <span>{t('Administration')}</span>
                        <ChevronDown className="ml-auto transition-transform group-data-[state=open]/collapsible:rotate-180" />
                      </SidebarMenuButton>
                    </CollapsibleTrigger>
                    <CollapsibleContent>
                      <SidebarMenuSub>
                        {adminItems.map((item) => (
                          <SidebarMenuSubItem key={item.title}>
                            <SidebarMenuSubButton isActive={location.pathname === item.path} asChild>
                              <Link to={item.path}>
                                <span>{item.title}</span>
                              </Link>
                            </SidebarMenuSubButton>
                          </SidebarMenuSubItem>
                        ))}
                      </SidebarMenuSub>
                    </CollapsibleContent>
                  </SidebarMenuItem>
                </Collapsible>
              )}
            </SidebarMenu>
          </SidebarGroupContent>
        </SidebarGroup>
      </SidebarContent>
      <SidebarFooter>
          <SidebarMenu>
            <SidebarMenuItem>
              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <SidebarMenuButton size="lg" className="group/profile">
                    <div className="bg-linear-to-bl from-primary/50 to-primary size-8 rounded-full relative shrink-0 overflow-hidden">
                      <div className="absolute bg-linear-to-t from-transparent via-background dark:via-foreground to-transparent size-full rotate-45 -translate-x-full translate-y-full group-hover/profile:translate-x-full group-hover/profile:-translate-y-full transition-transform duration-300 group-data-[state=open]/profile:translate-x-full group-data-[state=open]/profile:-translate-y-full" />
                    </div>
                    <div className="flex flex-col">
                      {user?.username || t('Username')}
                      {productInfo?.version && (
                        <span className="block text-muted-foreground text-xs">
                          {productInfo.version}
                        </span>
                      )}
                    </div>
                    <ChevronsUpDown className="ml-auto" />
                  </SidebarMenuButton>
                </DropdownMenuTrigger>
                <DropdownMenuContent className="w-(--radix-dropdown-menu-trigger-width)">
                  <Link to='/profile'>
                    <DropdownMenuItem>
                      <span>{t('Profile')}</span>
                    </DropdownMenuItem>
                  </Link>
                  <DropdownMenuItem onClick={() => logoutMutation.mutate()}>
                    <span>{t('Log Out')}</span>
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            </SidebarMenuItem>
          </SidebarMenu>
        </SidebarFooter>
    </Sidebar>
  )
}