import { Navigate, Outlet } from 'react-router-dom';
import { useAuth } from '@/contexts/AuthContext';

type RouteProtectorProps =
  | { type: 'login' }
  | { type: 'install' }
  | { type: 'protected' }
  | { type: 'admin' }
  | { type: 'enterkey' }
  | { type: 'protected' }
  ;

export default function RouteProtector({ type }: RouteProtectorProps) {
  const { 
    user, 
    installed, 
    isUserLoading, 
    isInstalledLoading,
    trial,
    isTrialLoading,
    license,
    isLicenseLoading,
  } = useAuth();

  if (isUserLoading || isInstalledLoading || isTrialLoading || isLicenseLoading || !license || !trial) return <div/>;

  if (!installed && type !== 'install') return <Navigate to="/install" replace />;
  if (installed && type === 'install') return <Navigate to="/login" replace />;

  // Redirect /login to main if already logged in
  if (type === 'login' && user) {
    return <Navigate to={`/`} replace />;
  }

  // Protected routes require a user
  if (type === 'protected' && !user) {
    return <Navigate to="/login" replace />;
  }

  // Redirect non-admins to login or 404 for admin routes
  if (type === 'admin') {
    if (!user) return <Navigate to="/login" replace />;
    if (!user.is_admin) return <Navigate to="/404" replace />;
  }
  
  // Redirect /enterkey to main if trial active or license exists
  if (type === 'enterkey' && (!trial.trial_expired || license.key)) {
    return <Navigate to="/" replace />;
  }

  // Redirect to /enterkey page if trial is expired and no valid license
  if (type === 'protected' && trial.trial_expired && !license.key) {
    return <Navigate to={`/enterkey`} replace />;
  }

  return <Outlet />;
}
