import {
    FormControl,
    FormDescription,
    FormField,
    FormItem,
    FormLabel,
} from "@/components/ui/form";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
    Select,
    SelectTrigger,
    SelectValue,
    SelectContent,
    SelectItem
} from "@/components/ui/select";

import { useTranslation } from "react-i18next";
import { useState,  } from "react";
import { AuthServer, FormValueSignin  } from "@/components/administration/types";
import { Card, CardContent } from "../ui/card";
import { Eye, EyeOff } from "lucide-react";
import { useGetAuthServers } from "./queries";
import { UseFormReturn } from "react-hook-form";



// Define props for ImportUserStep1
interface ImportUserStep1Props {
    form: UseFormReturn<FormValueSignin>;
}

// New ImportUserStep1 component
export default function ImportUserStep1({ form }: ImportUserStep1Props) {
    const { t } = useTranslation();
    const [showPassword, setShowPassword] = useState(false);
    const { data: authServers, isLoading } = useGetAuthServers();
    //const [serverOptions] = useState<AuthServer[]>(authServers);

    const getServerLabel = (server: AuthServer) => {
        if (server.type.toLowerCase() === "ldap") {
            return `${t('LDAP - ')} ${server.host}`;
        }
        return `${t('Active Directory - ')} ${server.controllers}`;
    };

    if (isLoading) return <div className="h-[400px] flex justify-center items-center">{t('Loading')}...</div>

    return (
        <Card className="w-4/5 items-center mx-auto mb-8 mt-4 p-4 max-h-[550px]">
            <CardContent className="px-0">
                <div className="space-y-6">
                    {/* Server Dropdown */}
                    <FormField
                        control={form.control}
                        name="server"
                        render={({ field }) => (
                            <FormItem>
                                <FormLabel className="gap-0">
                                    {t("Server")} <span className="text-error">*</span>
                                </FormLabel>
                                <FormControl>
                                    <Select
                                        value={field.value?.id?.toString() || ""}
                                        onValueChange={(value) => field.onChange(authServers.find((option: AuthServer) => option.id.toString() === value) || null)}
                                    >
                                        <SelectTrigger>
                                            <SelectValue placeholder={t("Select a server")} />
                                        </SelectTrigger>
                                        <SelectContent>
                                            {authServers.map((option: AuthServer) => (
                                                <SelectItem key={option.id} value={option.id.toString()}>
                                                    {getServerLabel(option)}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </FormControl>
                                <FormDescription>
                                    {t("Select the authentication server you previously configured")}
                                </FormDescription>
                            </FormItem>
                        )}
                    />

                    {/* Username */}
                    <FormField
                        control={form.control}
                        name="username"
                        render={({ field }) => (
                            <FormItem>
                                <FormLabel className="gap-0">
                                    {t("Username")}<span className="text-error">*</span>
                                </FormLabel>
                                <FormControl>
                                    <Input {...field} placeholder={t("Username")} />
                                </FormControl>
                                <FormDescription>
                                    {t("Admin credentials are only used to browse directory contents, not stored")}
                                </FormDescription>
                            </FormItem>
                        )}
                    />

                    {/* Password */}
                    <FormField
                        control={form.control}
                        name="password"
                        render={({ field }) => (
                            <FormItem>
                                <FormLabel className="gap-0">
                                    {t("Password")} <span className="text-error">*</span>
                                </FormLabel>
                                <FormControl>
                                    <div className="relative">
                                        <Input
                                            type={showPassword ? "text" : "password"}
                                            {...field}
                                            placeholder={t("Password")}
                                            className="pr-10"
                                        />
                                        <Button
                                            type="button"
                                            variant="ghost"
                                            size="icon"
                                            className="absolute inset-y-0 right-0"
                                            onClick={() => setShowPassword(!showPassword)}
                                        >
                                            {showPassword ? <EyeOff /> : <Eye />}
                                        </Button>
                                    </div>
                                </FormControl>
                            </FormItem>
                        )}
                    />
                </div>
            </CardContent>
        </Card>
    );
}