import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import axios from "axios";
import { useTranslation } from "react-i18next";
import { toast } from "sonner";
import { AuthServer, FormValueSignin, ImportUser, ConfigOptions, AuthValidationErrorResponse, BaseBackup } from "./types";
import { AxiosError } from "axios";

// Fetch current config
export function useGetConfigOptions() {
    return useQuery<ConfigOptions>({
        queryKey: ["global"],
        queryFn: async () => {
            const response = await axios.get("/api/v1/administration/global");
            return response.data;
        },
    });
}

// Update config
export function useUpdateConfigOptions() {
    const queryClient = useQueryClient();
    const { t } = useTranslation();

    return useMutation({
        mutationFn: (values: ConfigOptions) =>
            axios.put("/api/v1/administration/global", values),
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["global"] });
            localStorage.removeItem('locale');
            queryClient.invalidateQueries({ queryKey: ["language"] });
            queryClient.invalidateQueries({ queryKey: ["theme"] });
            toast(t("Configuration updated successfully"));
        },
        onError: () => {
            toast(t("Failed to update configuration"));
        },
    });
}

export function useGetAuthServers() {
    return useQuery({
        queryKey: ["auth_server"],
        queryFn: async () => {
            const res = await axios.get("/api/v1/auth_servers");
            return res.data;
        },
    });
}


export function usePutAuthServers () {
    const queryClient = useQueryClient();
    const { t } = useTranslation();

    return useMutation({
        mutationFn: (data : AuthServer) => axios.put("/api/v1/auth_servers", data),
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["auth_server"] });
            toast.success(t("Authentication server updated successfully"));
        },
        onError: () => {
            toast.error(t("Failed to update authentication server"));
        },
    });
}

export function usePostAuthServers () {
    const queryClient = useQueryClient();
    const { t } = useTranslation();

    return useMutation({
        mutationFn: (data : AuthServer) => axios.post('/api/v1/auth_servers', data),
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["auth_server"] });
            toast.success(t("Authentication server created successfully"));
        },
        onError: () => {
            toast.error(t("Failed to create authentication server"));
        },
    });
}

export function useDeleteAuthServers () {
    const queryClient = useQueryClient();
    const { t } = useTranslation();

    return useMutation({
        mutationFn: async (id: number) => {
            return await axios.delete(`/api/v1/auth_servers/${id}`);
        },
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["auth_server"] });
            toast.success(t("Authentication server deleted successfully"));
        },
        onError: () => {
            toast.error(t("Failed to delete authentication server"));
        },
    });
}


export function useValidateAuthServers() {
    const queryClient = useQueryClient();
    const { t } = useTranslation();


    return useMutation({
        mutationFn: (data: FormValueSignin) => axios.post('/api/v1/auth_servers/validate', data),
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["auth_server"] });
        },
        onError: (error: AxiosError) => {
            
            const status = error.response?.status;
            const errorData = error.response?.data;
            const errorCode = (errorData as AuthValidationErrorResponse)?.error_code ?? null;
            
            switch (status) {
                case 401:
                    // Handle different types of authentication failures
                    switch (errorCode) {
                        case 'AUTH_FAILED':
                            toast.error(t("Incorrect username or password"));
                            break;
                        case 'PASSWORD_EXPIRED':
                            toast.error(t("Password has expired and must be changed"));
                            break;
                        case 'ACCOUNT_DISABLED':
                            toast.error(t("Account is disabled"));
                            break;
                        case 'ACCOUNT_EXPIRED':
                            toast.error(t("Account has expired"));
                            break;
                        case 'ACCOUNT_LOCKED':
                            toast.error(t("Account is locked out"));
                            break;
                        default:
                            toast.error(t("Authentication failed"));
                    }
                    break;
                    
                case 495:
                    // SSL Certificate Error
                    if (errorCode === 'SSL_CERT_ERROR') {
                        toast.error(t("SSL certificate validation failed. Please verify your certificate authority is trusted or disable SSL certificate verification."));
                    } else if (errorCode === 'TLS_ERROR') {
                        toast.error(t("TLS connection failed. Please check your encryption settings."));
                    } else {
                        toast.error(t("SSL/TLS connection error. Please check your security settings."));
                    }
                    break;
                    
                case 502:
                    // Connection or LDAP errors
                    if (errorCode === 'CONNECTION_ERROR') {
                        toast.error(t("Unable to connect to the server. Please check the hostname, port, and network connectivity."));
                    } else if (errorCode === 'LDAP_ERROR') {
                        toast.error(t("LDAP connection error. Please verify your server configuration."));
                    } else {
                        toast.error(t("Unable to connect to the authentication server. Please check your connection settings."));
                    }
                    break;
                    
                case 422:
                    // Invalid request data
                    toast.error(t("Please fill in all required fields."));
                    break;
                    
                case 500:
                    // Server errors
                    toast.error(t("Server error occurred. Please try again later."));
                    break;
                    
                default:
                    // Fallback for any other errors
                    const message = (errorData as AuthValidationErrorResponse)?.message || 'Failed to validate credentials';
                    toast.error(t(message));
            }
        },
    });
}

export function useQueryAuthServer(authServerInfo: FormValueSignin, searchInput: string , searchParam: string) {
    return useQuery({
        queryKey: ['tree'],
        queryFn: async () => {
            const res = await axios.post('/api/v1/auth_servers/query', {
                ...authServerInfo,
                query: searchInput,
                param: searchParam, 
            });
            return res.data.hierarchy;
        },
    });
}

export function useImportUsers () {
    const queryClient = useQueryClient();
    const { t } = useTranslation();

    return useMutation({
        mutationFn: (data : ImportUser[]) => axios.post('/api/v1/auth_servers/import', data),
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["auth_server"] });
            toast.success(t("Successfully imported users"))
        },
        onError: (error: AxiosError) => {
             if (error.response?.status === 422) { //This displays the input error via toast
                const data = error.response.data as {
                    errors: Record<string, string[]>;
                };

                const apiErrors = data.errors;
                if (apiErrors) {
                    const entries = Object.entries(apiErrors);
                    const formattedErrors = entries.map(([index, messages]) => {
                        const displayIndex = Number(index) + 1;
                        return `User ${displayIndex}: ${(messages as string[]).join(" | ")}`;
                    });
                    toast.error(formattedErrors.join(" ,"), { duration: 8000 });
                } 
                else {
                    toast.error(t("One or more of your usernames/emails are invalid"));
                }
            } 
            else {
                toast.error(t("Authentication server user creation failed"));
            }
        },
    });
}

export function useGetCertificates(){
    return useQuery({
        queryKey: ["certificates"],
        queryFn: async () => {
            const res = await axios.get("/api/v1/certificates");
            return res.data;
        },
    });
}

export function usePostCertificates(){
    const queryClient = useQueryClient();
    const { t } = useTranslation();
    return useMutation({
        mutationFn: (data: { certificate: string }) => axios.post("/api/v1/certificates", data),
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["certificates"] });
            toast.success(t("Certificate successfully added"));
        },
        onError: (error: AxiosError) => {
            if (error.response?.status === 409) {
                toast.error(t("Certificate already exists"));
            } else {
                toast.error(t("Failed to add certificate"));
            }
        },
    });
}

export function useDeleteCertificates(){
    const queryClient = useQueryClient();
    const { t } = useTranslation();

    return useMutation({
        mutationFn: async (id: number) => {
            return await axios.delete(`/api/v1/certificates/${id}`);
        },
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["certificates"] });
            toast.success(t("Certificate deleted successfully"));
        },
        onError: () => {
            toast.error(t("Failed to delete certificate"));
        },
    });
}

export function useExtractCN() {
    const { t } = useTranslation();

    return useMutation({
        mutationFn: async (certificate: string) => {
            const res = await axios.post("/api/v1/certificates/parse", { certificate });
            return res.data.hostname;
        },
        onError: () => {
            toast.error(t("Failed to extract CN from certificate"));
        },
    });
}

export function useGetBackups() {
    return useQuery<BaseBackup[], AxiosError>({
        queryKey: ["backups"],
        queryFn: async () => {
            const response = await axios.get("/api/v1/administration/backup");
            return response.data;
        },
    });
}

export function useCreateBackup() {
    const { t } = useTranslation();

    return useMutation({
        mutationFn: () => axios.post('/api/v1/administration/backup'),
        onSuccess: () => {
            toast.success(t("Backup request successfully sent"));
        },
        onError: () => {
            toast.error(t("Failed to create backup"));
        },
    });
}

export function useDownloadBackup() {
    const { t } = useTranslation();
    return useMutation({
        mutationFn: async (file_name: string) => {
            const response = await axios.get(`/api/v1/administration/backup/${file_name}/download`, {
                responseType: "blob",
            });

            const url = window.URL.createObjectURL(new Blob([response.data]));
            const link = document.createElement("a");
            link.href = url;

            link.setAttribute("download", `${file_name}`);
            document.body.appendChild(link);
            link.click();
            link.remove();
        },
        onError: () => {
            toast.error(t("Failed to download backup"));
        },
    });
}

export function useDeleteBackup() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: (file_name: string) => axios.delete(`/api/v1/administration/backup/${file_name}`),
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ["backups"] });
            toast(t("Backup deleted."));
        },
        onError: () => {
            toast(t("Failed to delete backup."), {
                description: t("Please try again"),
            });
        },
    });
}
