import { AlertingType, ServiceHostname, SNMPReceiver, Command } from "./types";
import { UseFormReturn } from "react-hook-form";
import { TFunction } from "i18next";
import { FormItem, FormControl, FormLabel, FormMessage, FormField } from "@/components/ui/form";
import { Checkbox } from "@/components/ui/checkbox";
import { Loader2Icon } from "lucide-react";
import { useGetUsers } from "../user/queries";
import { useGetServiceHostnames, useGetSNMPReceivers, useGetCommands } from "./queries";
import { User } from "../user/types";

export default function CheckReceiverSelector({ 
  alertingType, 
  form, 
  t, 
  step 
}: { 
  alertingType: AlertingType, 
  form: UseFormReturn<any>, // eslint-disable-line @typescript-eslint/no-explicit-any
  t: TFunction, 
  step: 'step2' | 'step3'
}) {
  const { data: emailRecipients, isLoading: emailRecipientsLoading } = useGetUsers();
  const { data: nagiosRecipients, isLoading: nagiosRecipientsLoading } = useGetServiceHostnames();
  const { data: snmpRecipients, isLoading: snmpRecipientsLoading } = useGetSNMPReceivers();
  const { data: commandRecipients, isLoading: commandRecipientsLoading } = useGetCommands();

  let alertingRecipients: ServiceHostname[] | SNMPReceiver[] | Command[] | User[] = [];
  if (alertingType === 'user') {
    alertingRecipients = emailRecipients as User[];
  } else if (alertingType === 'nagios') {
    alertingRecipients = nagiosRecipients as ServiceHostname[];
  } else if (alertingType === 'snmp_receiver') {
    alertingRecipients = snmpRecipients as SNMPReceiver[];
  } else if (alertingType === 'command') {
    alertingRecipients = commandRecipients as Command[];
  }

  return (
    <div className="flex flex-col gap-4 w-full min-h-64 border rounded-md p-4">
      <FormItem className="flex flex-col gap-4">
        {
          emailRecipientsLoading || nagiosRecipientsLoading || snmpRecipientsLoading || commandRecipientsLoading ? (
            <div className="flex justify-center items-center h-full">
              <Loader2Icon className="animate-spin" />
            </div>
          ) : (
            <>
              {
                alertingRecipients.length > 0 ? (
                  <>
                    {
                      alertingRecipients.map((recipient: ServiceHostname | SNMPReceiver | Command | User) => (
                        <FormField
                          key={recipient.id}
                          control={form.control}
                          name={`${step}.${alertingType}`}
                          render={({ field }) => {
                            return (
                              <FormItem
                                key={recipient.id}
                                className="flex flex-row items-center gap-2"
                              >
                                <FormControl>
                                  <Checkbox
                                    checked={field.value?.includes(recipient.id)}
                                    onCheckedChange={(checked) => {
                                      return checked
                                        ? field.onChange([...(field.value || []), recipient.id])
                                        : field.onChange(
                                          field.value?.filter(
                                            (value: number) => value !== recipient.id
                                          )
                                        )
                                    }}
                                  />
                                </FormControl>
                                <FormLabel className="text-sm font-normal">
                                  {RecieverLabelText({ recipient, alertingType })}
                                </FormLabel>
                              </FormItem>
                            )
                          }}
                        />
                      ))
                    }
                  </>
                ) : (
                  <div className="flex justify-center items-center h-full">
                    <p className="text-sm text-muted-foreground">{t(`No results.`)}</p>
                  </div>
                )
              }
            </>
          )
        }
        <FormMessage />
      </FormItem>
    </div>
  );
};

const RecieverLabelText = ({ recipient, alertingType }: { recipient: ServiceHostname | SNMPReceiver | Command | User, alertingType: AlertingType }) => {
  if (alertingType === 'nagios' && 'hostname' in recipient && 'servicename' in recipient && 'nagios_server' in recipient) {
    return `${recipient.hostname}/${recipient.servicename} on ${recipient.nagios_server.name}`;
  } else if (alertingType === 'user' && 'username' in recipient) {
    return recipient.username;
  } else if (alertingType === 'snmp_receiver' && 'name' in recipient) {
    return recipient.name;
  } else if (alertingType === 'command' && 'name' in recipient) {
    return recipient.name;
  }
  return 'unknown';
}