import { Button } from '@/components/ui/button'
import { PlusIcon, SquareTerminalIcon } from 'lucide-react'
import { useState } from 'react'
import { AlertDialog, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger, AlertDialogAction } from '@/components/ui/alert-dialog'

import { z } from "zod"
import { useForm } from "react-hook-form"
import { FormField, FormItem, FormLabel, FormControl, FormMessage, Form } from "@/components/ui/form"
import { Input } from "@/components/ui/input"
import { zodResolver } from '@hookform/resolvers/zod'
import { CommandFormData } from '@/components/alerting/types'
import { useCreateCommand } from '@/components/alerting/queries'
import { useTranslation } from 'react-i18next'
import { useAuth } from '@/contexts/AuthContext'

const commandFormSchema = z.object({
  name: z.string().min(1, "Command name required").max(255, "Must be less than 255 characters"),
  location: z.string().min(1, "Command required"),
  script: z.string(),
  arguments: z.string().optional(),
})

export default function AddCommandModal() {
  const { t } = useTranslation();
  const { mutate: createCommand } = useCreateCommand();
  const [newCommandFormOpen, setNewCommandFormOpen] = useState(false)
  const { user } = useAuth();

  const form = useForm<z.infer<typeof commandFormSchema>>({
    resolver: zodResolver(commandFormSchema),
    mode: 'onBlur',
    defaultValues: {
      name: '',
      location: '',
      script: '',
      arguments: '',
    },
  });

  const onSubmit = (data: CommandFormData) => {
    createCommand(data)
    form.reset()
  }

  const handleFormOpen = (open: boolean) => {
    setNewCommandFormOpen(open)
  }

  const handleCancel = () => {
    handleFormOpen(false)
    form.reset()
  }

  return (
    <AlertDialog open={newCommandFormOpen} onOpenChange={handleFormOpen}>
      {!!user?.is_admin &&
        <AlertDialogTrigger asChild>
          <Button className="w-fit text-xs font-bold">
            <PlusIcon />
            {t(`New Command`)}
          </Button>
        </AlertDialogTrigger>
      }
      <AlertDialogContent className='flex flex-col min-w-[800px]'>
        <AlertDialogHeader>
          <AlertDialogTitle className="flex h-min w-full mb-2 items-center gap-2">
            <SquareTerminalIcon size={20} />
            <span className="text-lg font-medium">
              {t(`Add Command`)}
            </span>
          </AlertDialogTitle>
        </AlertDialogHeader>
        <AlertDialogDescription>
          {t(`Add a new command to NNA, allowing you to send alerts directly to a service/host on your Nagios server.`)}
        </AlertDialogDescription>

        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="flex flex-col gap-4">
            <FormField
              control={form.control}
              name="name"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Command Name`)}</FormLabel>
                  <FormControl>
                    <Input {...field} placeholder="Enter command name" />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="location"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Script Location`)}</FormLabel>
                  <FormControl>
                    <Input {...field} placeholder="Script location (e.g. /usr/bin)" />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="script"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Script Name`)}</FormLabel>
                  <FormControl>
                    <Input {...field} placeholder="Script name (e.g. notify.sh)" />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="arguments"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Passed Arguments`)}</FormLabel>
                  <FormControl>
                    <Input {...field} placeholder="Optional arguments" />
                  </FormControl>
                  <FormMessage />
                  <div className='flex flex-col gap-2 text-muted-foreground'>
                    {t(`You can pass some basic macros to the script via arguments that will be auto-populated when the script is executed.`)}
                    <ul className='list-disc list-inside'>
                      <li>{`%sourcename%`} - {t(`the name of the source that is being alerted on`)}</li>
                      <li>{`%sourcegroupname%`} - {t(`the name of the source group that is being alerted on`)}</li>
                      <li>{`%state%`} - {t(`the state of the check`)}</li>
                      <li>{`%returncode%`} - {t(`the return code of the check`)}</li>
                      <li>{`%output%`} - {t(`the output of the check`)}</li>
                    </ul>
                  </div>
                </FormItem>
              )}
            />

            {/* FOOTER */}
            <AlertDialogFooter>
              <AlertDialogCancel type='button' onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>
              <AlertDialogAction disabled={!form.formState.isValid || form.formState.isSubmitting} type="submit" className="w-fit">
                {t(`Add Command`)}
              </AlertDialogAction>
            </AlertDialogFooter>
          </form>
        </Form>
      </AlertDialogContent>
    </AlertDialog>
  )
}
