import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger } from '@/components/ui/alert-dialog'
import { Button } from '@/components/ui/button'
import { HardDriveIcon, PlusIcon } from 'lucide-react'
import { z } from 'zod'
import { zodResolver } from '@hookform/resolvers/zod'
import { useForm } from 'react-hook-form'
import { useState } from 'react'
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form'
import { Input } from '@/components/ui/input'
import { useCreateNagiosServer } from '@/components/alerting/queries'
import { NagiosServerFormData } from '@/components/alerting/types'
import { useTranslation } from 'react-i18next'
import { useAuth } from '@/contexts/AuthContext'

const nagiosServerFormSchema = z.object({
  name: z.string().min(1),
  nrdp_url: z.string().url(),
  nrdp_token: z.string().min(1),
})

export default function AddNagiosServerModal() {
  const { t } = useTranslation();
  const [newNagiosServerFormOpen, setNewNagiosServerFormOpen] = useState(false)
  const { mutate: createNagiosServer } = useCreateNagiosServer();
  const { user } = useAuth();

  const form = useForm<z.infer<typeof nagiosServerFormSchema>>({
    resolver: zodResolver(nagiosServerFormSchema),
    mode: 'onChange',
    defaultValues: {
      name: '',
      nrdp_url: '',
      nrdp_token: '',
    },
  })

  const handleSubmit = (data: NagiosServerFormData) => {
    createNagiosServer(data)
  }

  const handleFormOpen = (open: boolean) => {
    setNewNagiosServerFormOpen(open)
    if (open) {
      form.reset()
    }
  }

  const handleCancel = () => {
    handleFormOpen(false)
    form.reset()
  }

  return (
    <AlertDialog open={newNagiosServerFormOpen} onOpenChange={handleFormOpen}>
      {!!user?.is_admin &&
        <AlertDialogTrigger asChild>
          <Button className="w-fit text-xs font-bold">
            <PlusIcon />
            {t(`New Nagios Server`)}
          </Button>
        </AlertDialogTrigger>
      }
      <AlertDialogContent>
        <AlertDialogHeader>
          <AlertDialogTitle className="flex h-min w-full mb-2 items-center gap-2">
            <HardDriveIcon size={20} />
            <span className="text-lg font-medium">
              {t(`Add Nagios Server`)}
            </span>
          </AlertDialogTitle>
        </AlertDialogHeader>
        <AlertDialogDescription>
          {t(`Add a new Nagios server to NNA, allowing you to send alerts directly to your Nagios server.`)}
        </AlertDialogDescription>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleSubmit)} className="flex flex-col gap-4 grow min-h-0">
            <FormField
              control={form.control}
              name="name"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Name`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="nrdp_url"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`NRDP URL`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="nrdp_token"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`NRDP Token`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            {/* FOOTER */}
            <AlertDialogFooter>
              <AlertDialogCancel onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>
              <AlertDialogAction disabled={!form.formState.isValid || form.formState.isSubmitting} type="submit" className="w-fit">
                {t(`Add Nagios Server`)}
              </AlertDialogAction>
            </AlertDialogFooter>
          </form>
        </Form>
      </AlertDialogContent>
    </AlertDialog>
  )
}
