import { Form, FormField, FormLabel, FormItem, FormControl, FormMessage } from "@/components/ui/form";
import { ClipboardCheckIcon, PencilIcon } from "lucide-react";
import { useTranslation } from "react-i18next";
import { z } from "zod";
import { zodResolver } from "@hookform/resolvers/zod";
import { useForm, UseFormReturn } from "react-hook-form";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { AlertDialog, AlertDialogTitle, AlertDialogHeader, AlertDialogContent, AlertDialogTrigger, AlertDialogDescription, AlertDialogFooter, AlertDialogCancel } from "@/components/ui/alert-dialog";
import { useEffect, useState } from "react";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { TFunction } from "i18next";
import { useUpdateCheck } from "@/components/alerting/queries";
import { AlertingType, SuricataCheck } from "../../types";
import { Tabs, TabsList, TabsTrigger, TabsContent } from "@/components/ui/tabs";
import { DropdownMenuItem } from "@/components/ui/dropdown-menu";
import CheckReceiverSelector from "../../CheckReceiverSelector";

const suricataCheckFormSchema = z.object({
  check_type: z.literal('suricata'),
  step1: z.object({
    name: z.string().min(1, "Check Name Required").max(255, "Must be less than 255 characters"),
    association: z.object({
      type: z.enum(['signature_id']),
      id: z.string(),
    }).superRefine((data, ctx) => {
      if (!data.id) {
        ctx.addIssue({
          code: z.ZodIssueCode.custom,
          message: "Signature ID Required",
          path: ["id"],
        });
      }
      if (parseInt(data.id) <= 0) {
        ctx.addIssue({
          code: z.ZodIssueCode.custom,
          message: "Signature ID must be a positive number",
          path: ["id"],
        });
      }
    }),
    metric: z.enum(['alert_count']),
    warning_threshold: z.string().min(1, "Warning Threshold Required"),
    critical_threshold: z.string().min(1, "Critical Threshold Required"),
    lookback_period: z.string().min(1, "Lookback Period Required"),
    check_frequency: z.string().min(1, "Check Frequency Required"),
  }),
  step2: z.object({
    user: z.array(z.number()),
    nagios: z.array(z.number()),
    snmp_receiver: z.array(z.number()),
    command: z.array(z.number()),
  }),
})

export default function EditSuricataCheckModal({ check, onModalClose }: { check: SuricataCheck, onModalClose: () => void }) {
  const { t } = useTranslation();
  const { mutate: updateCheck } = useUpdateCheck();

  const [newSuricataCheckFormOpen, setNewSuricataCheckFormOpen] = useState(false);
  const [formStep, setFormStep] = useState<'step1' | 'step2' | 'step3'>('step1');
  const [formStepValid, setFormStepValid] = useState<{ [key in 'step1' | 'step2']: boolean }>({
    step1: false,
    step2: false,
  });


  const form = useForm<z.infer<typeof suricataCheckFormSchema>>({
    resolver: zodResolver(suricataCheckFormSchema),
    mode: 'onBlur',
    defaultValues: {
      check_type: check.check_type,
      step1: {
        name: check.name,
        association: {
          type: 'signature_id',
          id: check.object_id.toString(),
        },
        metric: check.metric,
        warning_threshold: check.warning,
        critical_threshold: check.critical,
        lookback_period: getSuricataCheckQueryParam(check.raw_query, 'lookback_period') || '1d',
        check_frequency: getSuricataCheckQueryParam(check.raw_query, 'frequency') || '5min',
      },
      step2: {
        user: check.alerting_associations?.filter((association) => association.association_type === 'user').map((association) => association.association_id) || [],
        nagios: check.alerting_associations?.filter((association) => association.association_type === 'nagios').map((association) => association.association_id) || [],
        snmp_receiver: check.alerting_associations?.filter((association) => association.association_type === 'snmp_receiver').map((association) => association.association_id) || [],
        command: check.alerting_associations?.filter((association) => association.association_type === 'command').map((association) => association.association_id) || [],
      },
    },
  })

  // form step 1 validation handling
  useEffect(() => {
    if (form.formState.errors.step1) {
      return setFormStepValid((prev) => ({ ...prev, step1: false }));
    }
    return setFormStepValid((prev) => ({ ...prev, step1: true }));
  }, [form.formState.errors.step1]);

  // form step 2 validation handling
  useEffect(() => {
    if (form.formState.errors.step2) {
      return setFormStepValid((prev) => ({ ...prev, step2: false }));
    }
    return setFormStepValid((prev) => ({ ...prev, step2: true }));
  }, [form.formState.errors.step2]);

  // handle form open
  const handleFormOpen = (open: boolean) => {
    setNewSuricataCheckFormOpen(open);
    if (open) {
      setFormStep('step1');
      form.trigger();
    }
    if (!open) {
      onModalClose()
    }
    form.reset();
  }

  // handle movement between each step
  const handleStepping = ({ direction }: { direction: 'next' | 'previous' }) => {
    const steps = ['step1', 'step2'];
    const currentStepIndex = steps.indexOf(formStep);
    const nextStepIndex = direction === 'next' ? currentStepIndex + 1 : currentStepIndex - 1;
    const nextStep = steps[nextStepIndex];

    if (nextStep) {
      setFormStep(nextStep as 'step1' | 'step2');
    }
  };

  // handle submit
  const handleSubmit = (data: z.infer<typeof suricataCheckFormSchema>) => {
    updateCheck({ id: check.id, data });
    handleFormOpen(false);
  };

  // handle cancel
  const handleCancel = () => {
    setNewSuricataCheckFormOpen(false);
    setFormStep('step1');
    setFormStepValid({
      step1: false,
      step2: false,
    });
    form.reset();
  };

  return (
    <AlertDialog open={newSuricataCheckFormOpen} onOpenChange={handleFormOpen}>
      <AlertDialogTrigger asChild>
        <DropdownMenuItem onSelect={(e) => e.preventDefault()}>
          <PencilIcon />
          <span className="hidden lg:block">{t(`View/Edit`)}</span>
        </DropdownMenuItem>
      </AlertDialogTrigger>
      <AlertDialogContent className="flex flex-col max-h-[640px] min-w-[800px] gap-6 overflow-hidden">
        {/* HEADER */}
        <AlertDialogHeader className="flex flex-col h-min gap-0">
          <AlertingModalTitle formStep={formStep} t={t} />
          <AlertingModalDescription formStep={formStep} t={t} />
        </AlertDialogHeader>

        <Form {...form}>
          {/* {suricataChecks && ( */}
          <form onSubmit={form.handleSubmit(handleSubmit)} className="flex flex-col gap-4 grow min-h-0">
            {formStep === 'step1' && (
              <Step1 form={form} t={t} />
            )}
            {formStep === 'step2' && (
              <Step2 form={form} t={t} />
            )}

            {/* FOOTER */}
            <AlertDialogFooter>
              <AlertDialogCancel onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>

              {formStep !== 'step1' && (
                <Button type="button" variant="outline" onClick={() => handleStepping({ direction: 'previous' })}>
                  {t(`Previous`)}
                </Button>
              )}

              {formStep === 'step1' && (
                <Button disabled={!formStepValid.step1} className="w-20" type="button" onClick={() => handleStepping({ direction: 'next' })}>
                  {t(`Next`)}
                </Button>
              )}

              {formStep === 'step2' && (
                <Button
                  className="w-20"
                  type="submit"
                  disabled={!form.formState.isValid || form.formState.isSubmitting}
                >
                  {t(`Update`)}
                </Button>
              )}

            </AlertDialogFooter>
          </form>
          {/* )} */}
        </Form>

      </AlertDialogContent>
    </AlertDialog>
  )
}

const AlertingModalTitle = ({ formStep, t }: { formStep: 'step1' | 'step2' | 'step3', t: TFunction }) => {
  return (
    <AlertDialogTitle className="flex h-min w-full mb-2 items-center gap-2">
      <ClipboardCheckIcon size={20} />
      <span className="w-fit text-lg font-medium">
        {t(`Edit Suricata Check`)}
      </span>
      <span className="ml-auto text-base font-medium">
        {
          formStep === 'step1' ? (
            <>
              {t(`Step 1 - Select Criteria`)}
            </>
          ) : (
            <>
              {t(`Step 2 - Select Alerting`)}
            </>
          )
        }
      </span>
    </AlertDialogTitle >
  );
};

const AlertingModalDescription = ({ formStep, t }: { formStep: 'step1' | 'step2' | 'step3', t: TFunction }) => {
  return (
    <AlertDialogDescription>
      {
        formStep === 'step1' ? (
          <>
            {t(`Select the criteria you would like to check for.`)}
          </>
        ) : (
          <>
            {t(`Select how you would like to be notified of this check. Select any of the items in the lists under the tabs, and all the selected elements will be notified.`)}
          </>
        )
      }
    </AlertDialogDescription>
  );
};

const Step1 = ({ form, t }: { form: UseFormReturn<z.infer<typeof suricataCheckFormSchema>>, t: TFunction }) => {
  return (
    <div className="flex flex-col grow gap-8">
      <FormField
        control={form.control}
        name="step1.name"
        render={({ field }) => (
          <FormItem className="relative">
            <FormLabel>{t('Name')}</FormLabel>
            <FormControl>
              <Input {...field} />
            </FormControl>
            <FormMessage className="absolute -bottom-6" />
          </FormItem>
        )}
      />

      <div className="flex w-full flex-row gap-8">
        <FormField
          control={form.control}
          name="step1.association.type"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Check On')}</FormLabel>
              <FormControl>
                <Select
                  defaultValue={field.value}
                  onValueChange={(value) => field.onChange(value)}
                >
                  <SelectTrigger>
                    <SelectValue placeholder={t('Select a check on')} />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="signature_id">{t('Signature ID')}</SelectItem>
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="step1.association.id"
          render={({ field }) => (
            <FormItem className="w-full relative">
              <FormLabel>{t('Signature ID')}</FormLabel>
              <FormControl>
                <Input type="number" {...field} />
              </FormControl>
              <FormMessage className="absolute -bottom-6" />
            </FormItem>
          )}
        />
      </div>

      <div className="flex flex-row grow gap-8">
        <FormField
          control={form.control}
          name="step1.metric"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Metric')}</FormLabel>
              <FormControl>
                <Select
                  defaultValue={field.value}
                  onValueChange={(value) => field.onChange(value)}
                >
                  <SelectTrigger>
                    <SelectValue placeholder={t('Select a metric')} />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="alert_count">{t('Alert Count')}</SelectItem>
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <div className="flex w-full flex-row gap-4">
          <FormField
            control={form.control}
            name="step1.warning_threshold"
            render={({ field }) => (
              <FormItem className="w-full">
                <FormLabel>{t('Warning Threshold')}</FormLabel>
                <FormControl>
                  <Input {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          <FormField
            control={form.control}
            name="step1.critical_threshold"
            render={({ field }) => (
              <FormItem className="w-full">
                <FormLabel>{t('Critical Threshold')}</FormLabel>
                <FormControl>
                  <Input {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>
      </div>

      <div className="flex flex-row grow gap-8">
        <FormField
          control={form.control}
          name="step1.check_frequency"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Check Frequency')}</FormLabel>
              <FormControl>
                <Select
                  defaultValue={field.value}
                  onValueChange={(value) => field.onChange(value)}
                >
                  <SelectTrigger>
                    <SelectValue placeholder={t('Select a frequency')} />
                  </SelectTrigger>
                  <SelectContent>
                    {check_frequency_times.map((time) => (
                      <SelectItem key={time.value} value={time.value}>{t(time.label)}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="step1.lookback_period"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Lookback Period')}</FormLabel>
              <FormControl>
                <Select
                  defaultValue={field.value}
                  onValueChange={(value) => field.onChange(value)}
                >
                  <SelectTrigger>
                    <SelectValue placeholder={t('Select a lookback period')} />
                  </SelectTrigger>
                  <SelectContent>
                    {lookback_period_times.map((time) => (
                      <SelectItem key={time.value} value={time.value}>{t(time.label)}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
      </div>

    </div>
  )
}

const Step2 = ({ form, t }: { form: UseFormReturn<z.infer<typeof suricataCheckFormSchema>>, t: TFunction }) => {
  const alertingTypes: AlertingType[] = ['user', 'nagios', 'snmp_receiver', 'command'];

  return (
    <div className="flex flex-col gap-4 grow">
      <Tabs defaultValue={alertingTypes[0]}>
        <TabsList>
          {
            alertingTypes.map((type) => (
              <TabsTrigger key={type} value={type}>{t(type.split('_').map((word) => word.charAt(0).toUpperCase() + word.slice(1)).join(' '))}</TabsTrigger>
            ))
          }
        </TabsList>

        {
          alertingTypes.map((type) => (
            <TabsContent key={type} value={type}>
              <CheckReceiverSelector alertingType={type} form={form} step="step2" t={t} />
            </TabsContent>
          ))
        }

      </Tabs>
    </div>
  );
};

const check_frequency_times = [
  {
    value: '5min',
    label: 'Every 5 Minutes'
  },
  {
    value: 'Hourly',
    label: 'Hourly'
  },
  {
    value: 'Daily',
    label: 'Daily'
  },
  {
    value: 'Weekly',
    label: 'Weekly'
  },
  {
    value: 'Monthly',
    label: 'Monthly'
  },
  {
    value: 'Yearly',
    label: 'Yearly'
  },
]

const lookback_period_times = [
  {
    value: '1d',
    label: '1 Day'
  },
  {
    value: '7d',
    label: '7 Days'
  },
  {
    value: '30d',
    label: '30 Days'
  },
  {
    value: '90d',
    label: '90 Days'
  },
  {
    value: '180d',
    label: '180 Days'
  },
  {
    value: '365d',
    label: '365 Days'
  },
]

const getSuricataCheckQueryParam = (raw_query: string, param: string) => {
  const query_params = raw_query.split("&");
  const query_params_obj: { [key: string]: string } = {};
  query_params.forEach((param: string) => {
    const [key, value] = param.split("=");
    query_params_obj[key] = value;
  });
  return query_params_obj[param];
}