import { useMutation, useQuery } from "@tanstack/react-query";
import { useQueryClient } from "@tanstack/react-query";
import { useTranslation } from "react-i18next";
import { toast } from "sonner";
import axios from "axios";
import {
  CommandFormData,
  FlowSourceCheckFormData,
  NagiosServerFormData,
  NmapCheckFormData,
  ServiceHostnameFormData,
  SNMPReceiverFormData,
  SuricataCheckFormData,
} from "./types";
import {
  mockChecksData,
  mockNagiosServersData,
  mockAlertingServiceHostnamesData,
  mockCommandsData,
  mockNewCheckFormData,
  mockSNMPReceiversData,
} from "@/lib/mock/mock-alerting-data";


// check queries
export function useCreateCheck() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  
  return useMutation({
    mutationFn: (data: FlowSourceCheckFormData | NmapCheckFormData | SuricataCheckFormData) => {
      return axios.post('/api/v1/check', data);
    },
    onSuccess: (_, variables) => {
      queryClient.invalidateQueries({ queryKey: ['checks', variables.check_type] });
      toast(t('Check successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create check.'),
        { description: t('Please try again.') });
    },
  });
}
export function useUpdateCheck() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: ({ id, data }: { id: number, data: FlowSourceCheckFormData | NmapCheckFormData | SuricataCheckFormData }) => {
      return axios.put(`/api/v1/check/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['checks'] });
      toast(t('Check successfully updated.'));
    },
    onError: () => {
      toast(t('Failed to update check.'),
        { description: t('Please try again.') });
    },
  });
}
export function useGetChecks({ check_type }: { check_type: 'flow_source' | 'nmap' | 'suricata' }) {
  const USE_MOCK_DATA = false;


  return useQuery({
    queryKey: ['checks', check_type],
    queryFn: async () => {
      if (USE_MOCK_DATA) {
        return Promise.resolve(mockChecksData);
      }

      const response = await axios.get(`/api/v1/check?check_type=${check_type}`);
      return response.data;
    }
  });
}
export function useDeleteCheck() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => {
      return axios.delete(`/api/v1/check/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['checks'] });
      toast(t('Check successfully deleted.'));
    },
    onError: () => {
      toast(t('Failed to delete check.'),
        { description: t('Please try again.') });
    },
  });
}
export function useForceCheck() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => {
      return axios.get(`/api/v1/check/forceCheck/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['checks'] });
      toast(t('Running check now!'));
    },
    onError: () => {
      toast(t('Failed to force check.'),
        { description: t('Please try again.') });
    },
  });
}
export function useCreateTestCheck() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: () => {
      return axios.post('/api/v1/check', mockNewCheckFormData);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['checks'] });
      toast(t('Check successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create check.'),
        { description: t('Please try again.') });
    },
  });
}
// TODO: ADD OPTIMISTIC UPDATES TO THIS
export function useToggleActiveCheck() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => {
      return axios.put(`/api/v1/check/toggleActive/${id}`);
    },
    onSuccess: (res) => {
      queryClient.invalidateQueries({ queryKey: ['checks'] });
      toast(t(`Check successfully ${res.data.active ? "Activated" : "Deactivated"}.`));
    },
    onError: () => {
      toast(t(`Failed to update check active status.`),
        { description: t('Please try again.') });
    },
  });
}


// command queries
export function useCreateCommand() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: CommandFormData) => {
      return axios.post('/api/v1/command', data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['commands'] });
      toast(t('Command successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create command.'),
        { description: t('Please try again.') });
    },
  });
}
export function useGetCommands() {
  const USE_MOCK_DATA = false;

  return useQuery({
    queryKey: ['commands'],
    queryFn: () => {
      if (USE_MOCK_DATA) {
        return Promise.resolve(mockCommandsData);
      }
      return axios.get('/api/v1/command').then(res => res.data);
    },
  });
}
export function useUpdateCommand() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: ({ id, data }: { id: number, data: CommandFormData }) => {
      return axios.put(`/api/v1/command/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['commands'] });
      toast(t('Command successfully updated.'));
    },
    onError: () => {
      toast(t('Failed to update command.'),
        { description: t('Please try again.') });
    },
  });
}
export function useDeleteCommand() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: (id: number) => {
      return axios.delete(`/api/v1/command/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['commands'] });
      toast(t('Command successfully deleted.'));
    },
    onError: () => {
      toast(t('Failed to delete command.'),
        { description: t('Please try again.') });
    },
  });
}


// nagios server queries
export function useCreateNagiosServer() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: NagiosServerFormData) => {
      return axios.post('/api/v1/nagios_server', data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['nagios-servers'] });
      toast(t('Nagios server successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create Nagios server.'),
        { description: t('Please try again.') });
    },
  });
}
export function useGetNagiosServers() {
  const USE_MOCK_DATA = false;

  return useQuery({
    queryKey: ['nagios-servers'],
    queryFn: () => {
      if (USE_MOCK_DATA) {
        return Promise.resolve(mockNagiosServersData);
      }
      return axios.get('/api/v1/nagios_server').then(res => res.data);
    },
  });
}
export function useUpdateNagiosServer() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: ({ id, data }: { id: number, data: NagiosServerFormData }) => {
      return axios.put(`/api/v1/nagios_server/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['nagios-servers'] });
      toast(t('Nagios server successfully updated.'));
    },
    onError: () => {
      toast(t('Failed to update Nagios server.'),
        { description: t('Please try again.') });
    },
  });
}
export function useDeleteNagiosServer() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: (id: number) => {
      return axios.delete(`/api/v1/nagios_server/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['nagios-servers'] });
      toast(t('Nagios server successfully deleted.'));
    },
    onError: () => {
      toast(t('Failed to delete Nagios server.'),
        { description: t('Please try again.') });
    },
  });
}


// nagios server service/hostnames queries
export function useCreateServiceHostname() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: ServiceHostnameFormData) => {
      return axios.post('/api/v1/service_hostname', data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['service-hostnames'] });
      toast(t('Service/Hostname successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create Service/Hostname.'),
        { description: t('Please try again.') });
    },
  });
}
export function useGetServiceHostnames() {
  const USE_MOCK_DATA = false;

  return useQuery({
    queryKey: ['service-hostnames'],
    queryFn: () => {
      if (USE_MOCK_DATA) {
        return Promise.resolve(mockAlertingServiceHostnamesData);
      }
      return axios.get('/api/v1/service_hostname').then(res => res.data);
    },
  });
}
export function useUpdateServiceHostname() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: ({ id, data }: { id: number, data: ServiceHostnameFormData }) => {
      return axios.put(`/api/v1/service_hostname/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['service-hostnames'] });
      toast(t('Service/Hostname successfully updated.'));
    },
    onError: () => {
      toast(t('Failed to update Service/Hostname.'),
        { description: t('Please try again.') });
    },
  });
}
export function useDeleteServiceHostname() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation({
    mutationFn: (id: number) => {
      return axios.delete(`/api/v1/service_hostname/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['service-hostnames'] });
      toast(t('Service/Hostname successfully deleted.'));
    },
    onError: () => {
      toast(t('Failed to delete Service/Hostname.'),
        { description: t('Please try again.') });
    },
  });
}


// SNMP receiver queries
export function useCreateSNMPReceiver() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: SNMPReceiverFormData) => {
      return axios.post('/api/v1/snmp_receiver', data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['SNMP-receivers'] });
      toast(t('SNMP receiver successfully created.'));
    },
    onError: () => {
      toast(t('Failed to create SNMP receiver.'),
        { description: t('Please try again.') });
    },
  });
}
export function useGetSNMPReceivers() {
  const USE_MOCK_DATA = false;

  return useQuery({
    queryKey: ['SNMP-receivers'],
    queryFn: () => {
      if (USE_MOCK_DATA) {
        return Promise.resolve(mockSNMPReceiversData);
      }

      return axios.get('/api/v1/snmp_receiver').then(res => res.data);
    },
  });
}
export function useUpdateSNMPReceiver() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: ({ id, data }: { id: number, data: SNMPReceiverFormData }) => {
      return axios.put(`/api/v1/snmp_receiver/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['SNMP-receivers'] });
      toast(t('SNMP receiver successfully updated.'));
    },
    onError: () => {
      toast(t('Failed to update SNMP receiver.'),
        { description: t('Please try again.') });
    },
  });
}
export function useDeleteSNMPReceiver() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: number) => {
      return axios.delete(`/api/v1/snmp_receiver/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['SNMP-receivers'] });
      toast(t('SNMP receiver successfully deleted.'));
    },
    onError: () => {
      toast(t('Failed to delete SNMP receiver.'),
        { description: t('Please try again.') });
    },
  });
}