import { Link, useLocation } from 'react-router-dom';
import { CirclePlus } from "lucide-react";
import { Button } from "@/components/ui/button";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import { X } from 'lucide-react';
import { useGetDashboards } from "@/components/dashboard/queries";
import DeleteDashboardDialog from "@/components/dashboard/DeleteDashboardDialog";
import React, { useState } from 'react';
import AddEditDashboardDialog from "@/components/dashboard/AddEditDashboardDialog";
import { useTranslation } from 'react-i18next';
import { useDashboard } from "@/components/dashboard/DashboardContext";
import { useGetUsers } from "@/components/user/queries";
import { cn } from '@/lib/utils';

export default function DashboardDropdown({ children, className, align = "center" }: { children: React.ReactNode, className?: string, align?: "start" | "center" | "end" }) {

  const [open, setOpen] = useState(false);
  const [selectedUser, setSelectedUser] = useState<number | undefined>(undefined);
  const { data: dashboards, isLoading } = useGetDashboards();
  const { t } = useTranslation();
  const { dashboard } = useDashboard();
  const { data: users = [], isError: isUsersError } = useGetUsers();
  const location = useLocation();

  if (isLoading) return <div></div>;
  if (!dashboards || !users || !dashboard) return;

  const title = location.state?.title ? location.state.title : dashboard ? dashboard.title : "";
  const userId = selectedUser !== undefined ? selectedUser : location.state?.user ? location.state.user : dashboard.user_id;
  const currentDashboard = `${userId}-${title}`;
  const currentUser = users?.find((user) => user.id === userId)?.username;
  const usersAvailable = !isUsersError && users && users.length > 0;

  return (
    <div className={`${className} flex items-center justify-between gap-2`}>
      <Popover open={open} onOpenChange={setOpen}>
        <PopoverTrigger asChild>
          {children}
        </PopoverTrigger>
        <PopoverContent className={cn("grid grid-cols-2 w-fit p-0")} align={align}>
          {usersAvailable && (
            <Command defaultValue={currentUser} className="border-r rounded-tr-none rounded-br-none bg-background hover:bg-card transition-colors">
              <CommandInput placeholder={t("Search users...")} />
              <CommandList>
                <CommandEmpty>{t('No users found')}.</CommandEmpty>
                <CommandGroup heading={t("Users")}>
                  {users.map((user) => (
                    <CommandItem key={user.id} className="py-0" onSelect={() => setSelectedUser(user.id)} asChild>
                      <div className={cn("flex justify-between items-center w-full h-9 group hover:cursor-pointer data-[selected=true]:bg-accent/50", userId === user.id && "data-[selected=true]:bg-accent bg-accent")}>
                        <span className="truncate max-w-30">{user.username}</span>
                      </div>
                    </CommandItem>
                  ))}
                </CommandGroup>
              </CommandList>
            </Command>
          )}
          {/* Dashboard Command */}
          <Command defaultValue={currentDashboard} className={cn("bg-background hover:bg-card transition-colors", !usersAvailable && "col-span-2")}>
            <CommandInput placeholder={t("Search dashboards...")} />
            <CommandList>
              <CommandEmpty>{t('No dashboards found')}.</CommandEmpty>
              <AddEditDashboardDialog userId={userId}>
                <div className="text-xs flex gap-1 h-9 py-4 px-1 mx-1 mt-1 items-center hover:bg-secondary hover:cursor-pointer rounded-sm">
                  <Button variant="ghost" className="h-4 w-4 p-0 m-0">
                    <CirclePlus className="text-primary" />
                  </Button>
                  {t('New Dashboard')}
                </div>
              </AddEditDashboardDialog>
              <CommandGroup heading={t("Dashboards")}>
                {dashboards.filter((dashboard) => dashboard.user_id === userId).map((dashboard) => (
                  <CommandItem key={dashboard.id} className="py-0" value={`${dashboard.user_id}-${dashboard.title}`}>
                    <div className="flex justify-between items-center w-full h-9">
                      <Link state={{"title": dashboard.title, "icon": dashboard.icon, "user": selectedUser }} to={'/dashboards/' + dashboard.id} className="flex h-full w-full items-center gap-2" onClick={() => setOpen(false)}>
                        <span className="truncate max-w-30">{dashboard.title}</span>
                      </Link>
                      <DeleteDashboardDialog id={dashboard.id}>
                        <Button className="bg-transparent h-4 w-4 m-0 text-muted-foreground hover:bg-transparent group">
                          <X className="group-hover:text-white" />
                        </Button>
                      </DeleteDashboardDialog>
                    </div>
                  </CommandItem>
                ))}
              </CommandGroup>
            </CommandList>
          </Command>
        </PopoverContent>
      </Popover>
      <span className="text-2xl font-semibold truncate">{title}</span>
    </div>
  )
}