import { dashlet } from "@/components/dashboard/types";
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from 'axios';
import { useTranslation } from 'react-i18next';
import { toast } from "sonner";

export function useGetDashlets(dashboardId: number) {
  return useQuery<dashlet[], AxiosError>({
      queryKey: ['dashlets', dashboardId],
      queryFn: async () => {
          const response = await axios.get('/api/v1/dashlet', {
              params: {
                  dashboard_id: dashboardId,
              },
          });
          return response.data;
      },
      retry: false,
  });
}

export function useAddDashlet() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
      mutationFn: (data: dashlet) => {
          return axios.post(
              '/api/v1/dashlet',
              {
                  title: data.title,
                  description: data.description,
                  dashboard_id: data.dashboard_id,
                  data_options: data.data_options,
                  vis_options: data.vis_options,
                  width: data.width,
                  height: data.height,
                  x: data.x,
                  y: data.y,
              },
              {
                  headers: {
                      'content-type': 'application/x-www-form-urlencoded',
                  },
              }
          );
      },
      onSuccess: (_response, newData) => {
        queryClient.invalidateQueries({ queryKey: ['dashlets', newData.dashboard_id] });
        toast(t('Dashlet added.'));
      },
      onError: () => {
        toast(t('Failed to add dashlet.'),
          { description: t('Please try again.') });
      },
  });
}

export function useEditDashlet() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: ({ data }: { data: dashlet }) => {
            return axios.put(
                '/api/v1/dashlet',
                {
                    id: data.id,
                    dashboard_id: data.dashboard_id,
                    title: data.title,
                    description: data.description,
                    data_options: data.data_options,
                    vis_options: data.vis_options,
                },
                {
                    headers: {
                        'content-type': 'application/x-www-form-urlencoded',
                    },
                }
            );
        },
        onSuccess: (_response, newData) => {
            queryClient.invalidateQueries({ queryKey: ['dashlets', newData.data.dashboard_id] });
            toast(t('Dashlet edited.'));
        },
        onError: () => {
            toast(t('Failed to edit dashlet.'),
              { description: t('Please try again.') });
        },
    });
}

export function useDeleteDashlet() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: (data: { id: number, dashboard_id: number }) => {
            return axios.delete('/api/v1/dashlet', {
                params: {
                    ...data,
                },
            });
        },
        onMutate: async (newData) => {
            const key = ['dashlets', newData.dashboard_id];
            await queryClient.cancelQueries({ queryKey: key });
            const previousDashlets = queryClient.getQueryData(key);
            queryClient.setQueryData(key, (old: dashlet[]) => {
                const oldDashlets = old.filter((dashlet) => dashlet.id !== newData.id);
                return oldDashlets;
            });
            return { previousDashlets };
        },
        onSuccess: (_response, newData) => {
            queryClient.invalidateQueries({ queryKey: ['dashlets', newData.dashboard_id] });
            toast(t('Dashlet deleted.'));
        },
        onError: (_err, newData, context) => {
            queryClient.setQueryData(['dashlets', newData.dashboard_id], context?.previousDashlets)
            toast(t('Failed to delete dashlet.'),
                { description: t('Please try again.') });
        },
        onSettled: (_response, _err, newData) => {
            queryClient.invalidateQueries({ queryKey: ['dashlets', newData.dashboard_id] })
        },
    });
}