"use client"

import { Bar, BarChart, CartesianGrid, XAxis, YAxis } from "recharts"
import {
  ChartConfig,
  ChartContainer,
  ChartLegend,
  ChartLegendContent,
  ChartTooltip,
  ChartTooltipContent,
} from "@/components/ui/chart"
import { ChartColumn as Icon, LoaderCircle } from "lucide-react";
import { 
  useGetSourceSummaryData, 
  useGetSourceTopTalkersData, 
  useGetSourcesSummaryData,
  useGetNmapScheduledScanData,
} from "@/components/dashboard/dashlets/queries";
import { useTranslation } from "react-i18next";
import { DashletDataOptions } from "../types";
import NoDataAlert from "@/components/dashboard/dashlet/NoDataAlert";

export const meta = {
  name: 'bar',
  title: 'Bar Chart',
  description: 'A simple bar chart',
  icon: Icon
};

const defaultTalkerOptions = {
  source_id: 0,
  num_talkers: [5],
  order_by: "bytes",
  group_by: "srcip",
}

export default function ChartBar(props: DashletDataOptions) {
  
  const {
    data: singleSourceData,
    isLoading: singleSourceLoading,
    isError: singleSourceError,
  } = useGetSourceSummaryData(
    Number(props.single_source_data_options?.source_id) ?? 0,
    props.range,
    { enabled: props.type === "source" }
  );

  const {
    data: multiSourceData,
    isLoading: multiSourceLoading,
    isError: multiSourceError,
  } = useGetSourcesSummaryData(
    props.range,
    props.multi_source_data_options ?? { metric: "bytes", sources: [] },
    { enabled: props.type === "sources" }
  );

  const {
    data: talkersData,
    isLoading: talkersLoading,
    isError: talkersError,
  } = useGetSourceTopTalkersData(
    Number(props.top_talker_data_options?.source_id) ?? 0, 
    props.range, props.top_talker_data_options ?? defaultTalkerOptions, 
    { enabled: props.type === "talkers" }
  );

  const {
    data: singleScheduledScanData,
    isLoading: singleScheduledScanLoading,
    isError: singleScheduledScanError,
  } = useGetNmapScheduledScanData(
    Number(props.scheduled_scan_data_options?.scheduled_scan_id ?? 0),
    props.range,
    { enabled: props.type === "scheduledScan" }
  );

  let data: any = null;
  let isLoading = false;
  let isError = false;
  let dataKey = undefined;
  let nameKey = undefined;

  switch (props.type) {
    case "source":
      data = singleSourceData;
      isLoading = singleSourceLoading;
      isError = singleSourceError;
      dataKey = "bytes";
      nameKey = "date";
      break;
    case "sources":
      data = multiSourceData;
      isLoading = multiSourceLoading;
      isError = multiSourceError;
      dataKey = "bytes";
      nameKey = "date";
      break;
    case "scheduledScan":
      data = singleScheduledScanData;
      isLoading = singleScheduledScanLoading;
      isError = singleScheduledScanError;
      dataKey = props.scheduled_scan_data_options?.metrics[0];
      nameKey = "date";
      break;
    case "talkers":
      data = talkersData;
      isLoading = talkersLoading;
      isError = talkersError;
      dataKey = props.top_talker_data_options?.order_by ?? "bytes";
      nameKey = props.top_talker_data_options?.group_by ?? "srcip";
      break;
    default:
      data = null;
      isLoading = false;
      isError = false;
      break;
  }

  const { t } = useTranslation();

  const sourceDisplayArgs = [
    { key: "bytes", label: "Bytes" },
    { key: "flows", label: "Flows" },
    { key: "packets", label: "Packets" },
    { key: "bps", label: "Bits/Sec" },
  ];

  const scheduledScanDisplayArgs = [
    { key: "hosts_up", label: "Hosts Up" },
    { key: "ports_open", label: "Ports Open" },
    { key: "ports_closed", label: "Ports Closed" },
    { key: "ports_filtered", label: "Ports Filtered" },
  ]

  if (isError) return <div>{t("Source not found")}</div>
  if (isLoading || !data) return <div className="size-full flex items-center justify-center"><LoaderCircle className="animate-spin" /></div>

  let chartData = [];
  let chartConfig: ChartConfig = {};
  if (props.type === "source" && props.single_source_data_options?.metrics) {
    chartConfig = props.single_source_data_options.metrics.reduce((config, metric) => {
      config[metric] = {
        label: sourceDisplayArgs.find(arg => arg.key === metric)?.label,
        color: `var(--chart-${sourceDisplayArgs.findIndex(arg => arg.key === metric) + 1})`,
      };
      return config;
    }, {} as ChartConfig);
    chartData = data.chart_data;
  } else if (props.type === "sources" && props.multi_source_data_options?.sources) {
    chartConfig = Object.entries(props.multi_source_data_options.sources).reduce((config, [_, sourceData]) => {
      config[sourceData.source_id] = {
        label: sourceData.name,
        color: `var(--chart-${sourceData.color})`,
      };
      return config;
    }, {} as ChartConfig);
    chartData = data.chart_data;
  } else if (props.type === "talkers") {
    const dataKey = props.top_talker_data_options?.order_by ?? "bytes";
    const nameKey = props.top_talker_data_options?.group_by ?? "srcip";

    if (talkersData && talkersData[nameKey]) {
      const sliced = talkersData[nameKey].slice(1);

      chartConfig = {};

      chartData = sliced.map((flow: { [key: string]: string }, index: number) => {
        const nameValue = flow[nameKey];
        const numericValue = Number(flow[dataKey]);
        const color = `var(--chart-${(index % 5) + 1})`;

        chartConfig[nameValue] = {
          label: nameValue,
          color,
        };

        return {
          [nameKey]: nameValue,
          [dataKey]: numericValue,
          fill: color,
        };
      });
    }
  } else if (props.type === "scheduledScan" && props.scheduled_scan_data_options?.metrics) {
    chartConfig = props.scheduled_scan_data_options.metrics.reduce((config, metric) => {
      config[metric] = {
        label: scheduledScanDisplayArgs.find(arg => arg.key === metric)?.label,
        color: `var(--chart-${scheduledScanDisplayArgs.findIndex(arg => arg.key === metric) + 1})`,
      };
      return config;
    }, {} as ChartConfig);

    chartData = data.chart_data;
  }

  if (chartData.length === 0) return <NoDataAlert />

  return (
    <ChartContainer
      config={chartConfig}
      className="aspect-auto h-full w-full"
    >
      <BarChart data={chartData}>
        {/* <defs>
          {Object.keys(chartConfig).map((key) => (
            <linearGradient key={key} id={`fill${key}`} x1="0" y1="0" x2="0" y2="1">
              <stop offset="5%" stopColor={`var(--color-${key})`} stopOpacity={0.8} />
              <stop offset="95%" stopColor={`var(--color-${key})`} stopOpacity={0.1} />
            </linearGradient>
          ))}
        </defs> */}
        <CartesianGrid vertical={false} />
        <XAxis
          dataKey={nameKey}
          tickLine={false}
          axisLine={false}
          tickMargin={8}
          minTickGap={32}
        />
        <YAxis
          tickLine={false}
          axisLine={false}
          tickMargin={8}
          tickFormatter={(value) =>
            value >= 1_000_000
              ? `${(value / 1_000_000).toFixed(1)}M`
              : value >= 1_000
              ? `${(value / 1_000).toFixed(1)}K`
              : value.toString()
          }
        />
        <ChartTooltip
          cursor={false}
          content={
            <ChartTooltipContent
              indicator="dot"
            />
          }
        />
        {props.type === "talkers" ? (
          dataKey ? (
            <Bar
              isAnimationActive={!props.isExporting}
              dataKey={dataKey}
              radius={2}
            />
          ) : null
        ) : (
          Object.keys(chartConfig).map((key) => (
            <Bar
              isAnimationActive={!props.isExporting}
              key={key}
              dataKey={key}
              fill={`var(--color-${key})`}
              radius={2}
            />
          ))
        )}
        <ChartLegend content={<ChartLegendContent />} />
      </BarChart>
    </ChartContainer>
  )
}
