"use client"

import { Line, LineChart, CartesianGrid, XAxis, YAxis } from "recharts"
import {
  ChartConfig,
  ChartContainer,
  ChartLegend,
  ChartLegendContent,
  ChartTooltip,
  ChartTooltipContent,
} from "@/components/ui/chart"
import { 
  useGetSourceSummaryData, 
  useGetSourcesSummaryData,
  useGetNmapScheduledScanData,
} from "@/components/dashboard/dashlets/queries";
import { useTranslation } from "react-i18next";
import { ChartSpline as Icon, LoaderCircle } from "lucide-react";
import { DashletDataOptions } from "../types";
import NoDataAlert from "@/components/dashboard/dashlet/NoDataAlert";

export const meta = {
  name: 'line',
  title: 'Line Chart',
  description: 'A simple line chart',
  icon: Icon,
};

export default function ChartLine(props: DashletDataOptions) {

  const {
    data: singleSourceData,
    isLoading: singleSourceLoading,
    isError: singleSourceError,
  } = useGetSourceSummaryData(
    Number(props.single_source_data_options?.source_id) ?? 0,
    props.range,
    { enabled: props.type === "source" }
  );

  const {
    data: multiSourceData,
    isLoading: multiSourceLoading,
    isError: multiSourceError,
  } = useGetSourcesSummaryData(
    props.range,
    props.multi_source_data_options ?? { metric: "bytes", sources: [] },
    { enabled: props.type === "sources" }
  );

  const {
    data: singleScheduledScanData,
    isLoading: singleScheduledScanLoading,
    isError: singleScheduledScanError,
  } = useGetNmapScheduledScanData(
    Number(props.scheduled_scan_data_options?.scheduled_scan_id ?? 0),
    props.range,
    { enabled: props.type === "scheduledScan" }
  );

  let data: any = null;
  let isLoading = false;
  let isError = false;

  switch (props.type) {
    case "source":
      data = singleSourceData;
      isLoading = singleSourceLoading;
      isError = singleSourceError;
      break;
    case "sources":
      data = multiSourceData;
      isLoading = multiSourceLoading;
      isError = multiSourceError;
      break;
    case "scheduledScan":
      data = singleScheduledScanData;
      isLoading = singleScheduledScanLoading;
      isError = singleScheduledScanError;
      break;
    default:
      data = null;
      isLoading = false;
      isError = false;
      break;
  }

  const { t } = useTranslation();

  const sourceDisplayArgs = [
    { key: "bytes", label: "Bytes" },
    { key: "flows", label: "Flows" },
    { key: "packets", label: "Packets" },
    { key: "bps", label: "Bits/Sec" },
  ];

  const scheduledScanDisplayArgs = [
    { key: "hosts_up", label: "Hosts Up" },
    { key: "ports_open", label: "Ports Open" },
    { key: "ports_closed", label: "Ports Closed" },
    { key: "ports_filtered", label: "Ports Filtered" },
  ]

  if (isError) return <div>{t("Source not found")}</div>
  if (isLoading || !data) return <div className="size-full flex items-center justify-center"><LoaderCircle className="animate-spin" /></div>

  let chartConfig = {} satisfies ChartConfig;
  if (props.type === "source" && props.single_source_data_options?.metrics) {
    chartConfig = props.single_source_data_options.metrics.reduce((config, metric) => {
      config[metric] = {
        label: sourceDisplayArgs.find(arg => arg.key === metric)?.label,
        color: `var(--chart-${sourceDisplayArgs.findIndex(arg => arg.key === metric) + 1})`,
      };
      return config;
    }, {} as ChartConfig);
  } else if (props.type === "sources" && props.multi_source_data_options?.sources) {
    chartConfig = Object.entries(props.multi_source_data_options.sources).reduce((config, [_, sourceData]) => {
      config[sourceData.source_id] = {
        label: sourceData.name,
        color: `var(--chart-${sourceData.color})`,
      };
      return config;
    }, {} as ChartConfig);
  } else if (props.type === "scheduledScan" && props.scheduled_scan_data_options?.metrics) {
    chartConfig = props.scheduled_scan_data_options.metrics.reduce((config, metric) => {
      config[metric] = {
        label: scheduledScanDisplayArgs.find(arg => arg.key === metric)?.label,
        color: `var(--chart-${scheduledScanDisplayArgs.findIndex(arg => arg.key === metric) + 1})`,
      };
      return config;
    }, {} as ChartConfig);
  }

  const chartData = data.chart_data;
  if (chartData.length === 0) return <NoDataAlert />;

  return (
    <ChartContainer
      config={chartConfig}
      className="aspect-auto h-full w-full"
    >
      <LineChart data={chartData}>
        <CartesianGrid vertical={false} />
        <XAxis
          dataKey="date"
          tickLine={false}
          axisLine={false}
          tickMargin={8}
          minTickGap={32}
          tickFormatter={(value) =>
            new Date(value).toLocaleTimeString("en-US", {
              hour: "numeric",
              minute: "2-digit",
              hour12: false,
            })
          }
        />
        <YAxis
          // @ts-ignore: Symlog scale works but Recharts ScaleType does not include "symlog" until v3
          scale={"symlog"}
          domain={["dataMin", "dataMax"]}
          tickLine={false}
          axisLine={false}
          tickMargin={8}
          tickFormatter={(value) =>
            value >= 1_000_000
              ? `${(value / 1_000_000).toFixed(1)}M`
              : value >= 1_000
              ? `${(value / 1_000).toFixed(1)}K`
              : value.toString()
          }
        />
        <ChartTooltip
          cursor={false}
          content={
            <ChartTooltipContent
              labelFormatter={(value) => {
                return new Date(value).toLocaleDateString("en-US", {
                  month: "short",
                  day: "numeric",
                })
              }}
              indicator="dot"
            />
          }
        />
        {Object.keys(chartConfig).map((key) =>
          <Line
            isAnimationActive={!props.isExporting}
            key={key}
            dataKey={key}
            fill={`var(--color-${key})`}
            stroke={`var(--color-${key})`}
          />
        )}
        <ChartLegend content={<ChartLegendContent />} />
      </LineChart>
    </ChartContainer>
  )
}
