import React, { useEffect, useRef, useState } from 'react';
import { CommandDialog, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList, CommandSeparator } from '@/components/ui/command';
import { CommandLoading } from 'cmdk';
import { Link, useNavigate } from 'react-router-dom';
import { useTranslation } from 'react-i18next';
import { TFunction } from 'i18next';
import { SuggestData, SearchDataType, SuggestReturn } from "@/components/header/types";
import { useSearchSuggest } from "@/components/header/queries";
import routes from "@/routes"
import { useQueryClient } from '@tanstack/react-query';
import { useAuth } from "@/contexts/AuthContext";

function addSearchElement(searchData: SearchDataType, element: SuggestReturn, t: TFunction<"translation", undefined>) {
  if (!Object.prototype.hasOwnProperty.call(searchData, t(element.category))) {
    searchData[t(element.category)] = [];
  }
  searchData[t(element.category)].push({
    url: element.url,
    value: t(element.value),
    label: t(element.label),
  });
  return searchData;
}

export const Search = () => {
  const queryClient = useQueryClient();
  const [open, setOpen] = useState(false);
  const [value, setValue] = useState('');
  const { t } = useTranslation();
  const navigate = useNavigate();
  const { user } = useAuth();
  const scrollRef = useRef<HTMLDivElement>(null);
  const shouldScrollRef = useRef(true);

  let searchData: SearchDataType = {};
  routes.forEach((route) => {
    if (route.isSearchable && route.path !== undefined && route.label !== undefined && route.category !== undefined && (!route.adminOnly || user?.is_admin)) {
      addSearchElement(searchData, {
        url: route.path,
        value: (route.value !== undefined) ? t(route.value) : t(route.label),
        label: route.label,
        category: route.category
      }, t);
    }
  });

  // Search backend for pages and parse it
  const suggestQuery = useSearchSuggest(value);
  if (suggestQuery.isSuccess) {
    suggestQuery.data.forEach((element) => {
      addSearchElement(searchData, element, t)
    });
  }

  const handleOpenDialog = (e: KeyboardEvent) => {
    if (e.key === 'k' && (e.metaKey || e.ctrlKey)) {
      e.preventDefault();
      setOpen(!open);
    }
  };

  const handleMessage = (event: MessageEvent<string>) => {
    if (event.origin !== window.location.origin) return;
    if (event.data === 'search') {
      setOpen(!open);
    }
  };

  useEffect(() => {
    if (open) {
      queryClient.invalidateQueries({ queryKey: ["search"] });
    }
    document.addEventListener('keydown', handleOpenDialog);
    window.addEventListener('message', handleMessage);
    return () => {
      document.removeEventListener('keydown', handleOpenDialog);
      window.removeEventListener('message', handleMessage);
    };
  }, [open]);

  return (
    <>
      <div className="relative flex items-center">
        <input
          type="text"
          placeholder={t('Search NNA')}
          onFocus={() => setOpen(true)}
          id="header-search-box"
          className="relative h-8 w-full justify-start rounded-md border bg-transparent px-4 py-2 text-sm font-medium text-muted-foreground shadow-sm transition-colors hover:bg-muted/70 hover:text-accent-foreground focus-visible:outline-none focus-visible:ring-1 focus-visible:ring-ring"
        />
        <kbd className="pointer-events-none absolute right-2 top-1/2 inline-flex h-5 -translate-y-1/2 select-none items-center gap-1 rounded border bg-muted px-1.5 font-mono text-[10px] font-medium text-muted-foreground opacity-100">
          {window.navigator.userAgent.includes('Macintosh') ? (
            <>
              <span className="text-xs">⌘</span>K
            </>
          ) : (
            <span className="text-xs">Ctrl K</span>
          )}
        </kbd>
      </div>
      <CommandDialog open={open} onOpenChange={setOpen}>
        <CommandInput
          placeholder={t('Search for a page...')}
          value={value}
          onKeyDown={(event) => {
            if (event.key === 'ArrowDown' || event.key === 'ArrowUp') {
              shouldScrollRef.current = true;
            } else {
              shouldScrollRef.current = false;
            }
          }}
          onValueChange={(input) => setValue(input)}
        />
        <CommandList
          ref={scrollRef}
          className="h-80"
          onWheel={() => (shouldScrollRef.current = true)}
          onScroll={() => {
            if (!shouldScrollRef.current) {
              // Taken from the scrollSelectedIntoView() function from https://github.com/pacocoursey/cmdk/blob/main/cmdk/src/index.tsx
              const item = scrollRef.current?.querySelector('[cmdk-item=""][aria-selected="true"]');
              if (item) {
                if (item.parentElement?.firstChild === item) {
                  // First item in Group, ensure heading is in view
                  item.closest('[cmdk-group=""]')?.querySelector('[cmdk-group-heading=""]')?.scrollIntoView({ block: 'nearest' });
                }
                // Ensure the item is always in view
                item.scrollIntoView({ block: 'nearest' });
              }
            }
            shouldScrollRef.current = true;
          }}
        >
          {suggestQuery.isLoading && <CommandLoading />}
          <CommandEmpty>{suggestQuery.isLoading ? '' : t('No results found.')}</CommandEmpty>
          {Object.entries(searchData).map(([key, value]: [string, Array<SuggestData>]) => {
            return (
              <React.Fragment key={key}>
                <CommandGroup heading={<div className="pt-1.5 text-accent-foreground">{key}</div>}>
                  {value.map((element: SuggestData) => {
                    return (
                      <Link key={element.url} className="w-full" to={element.url} onClick={() => setOpen(false)}>
                        <div className="rounded-sm">
                          <CommandItem
                            className="flex gap-2 hover:cursor-pointer"
                            value={element.value}
                            onSelect={() => {
                              setOpen(false);
                              const url = new URL(element.url, window.location.origin);
                              navigate(url.pathname + url.search + url.hash);
                            }}
                          >
                            <span>{element.label}</span>
                          </CommandItem>
                        </div>
                      </Link>
                    );
                  })}
                </CommandGroup>
                <CommandSeparator />
              </React.Fragment>
            );
          })}
        </CommandList>
      </CommandDialog>
    </>
  );
};
