import React, { createContext, useContext, useRef, useState } from 'react';
import { useQueryClient } from '@tanstack/react-query';
import { nmapScan } from '@/components/nmap/scans/types';
import { ndiff } from '@/components/nmap/ndiffs/types';
import { useEcho } from '@laravel/echo-react';

type NmapScanUpdateEvent = {
  id: number;
  status: string;
  started_at?: string;
  finished_at?: string;
};

type NdiffUpdateEvent = {
  id: number;
  status: string;
  started_at?: string;
  finished_at?: string;
};

type NmapIntegrationUpdatesContextType = {
  scanUpdates: Partial<nmapScan>[];
  mergeScan: (scan: nmapScan) => nmapScan;
  ndiffUpdates: Partial<ndiff>[];
  mergeNdiff: (ndiff: ndiff) => ndiff;
}

type NmapIntegrationUpdatesProviderProps = {
  children: React.ReactNode;
}

const NmapIntegrationUpdatesContext = createContext<NmapIntegrationUpdatesContextType | undefined>(undefined);

export const NmapIntegrationUpdatesProvider: React.FC<NmapIntegrationUpdatesProviderProps> = ({ children }) => {
  const queryClient = useQueryClient();

  const [scanUpdates, setScanUpdates] = useState<Partial<nmapScan>[]>([]);
  const scanUpdatesRef = useRef(scanUpdates);
  scanUpdatesRef.current = scanUpdates;

  const [ndiffUpdates, setNdiffUpdates] = useState<Partial<ndiff>[]>([]);
  const ndiffUpdatesRef = useRef(ndiffUpdates);
  ndiffUpdatesRef.current = ndiffUpdates;

  const mergeScan = (scan: nmapScan): nmapScan => {
    const update = scanUpdatesRef.current.slice().reverse().find(update => update.id === scan.id);
    if (!update) return scan;
    return { ...scan, ...update };
  };

  const mergeNdiff = (ndiff: ndiff): ndiff => {
    const update = ndiffUpdatesRef.current.slice().reverse().find(update => update.id === ndiff.id);
    if (!update) return ndiff;
    return { ...ndiff, ...update };
  }

  useEcho(
    `NNA.Admins`,
    'NmapScanCreated',
    () => {
      queryClient.invalidateQueries({ queryKey: ['nmapScans'] });
    }
  );

  useEcho(
    `NNA.Admins`,
    'NmapScanUpdate',
    (event: NmapScanUpdateEvent) => {
      setScanUpdates(prev => [...prev, event]);
      if (event.finished_at) {
        queryClient.invalidateQueries({ queryKey: ['nmapScans'] });
      }
    }
  );

  useEcho(
    `NNA.Admins`,
    'NdiffCreated',
    () => {
      queryClient.invalidateQueries({ queryKey: ['ndiffs'] });
    }
  );

  useEcho(
    `NNA.Admins`,
    'NdiffUpdate',
    (event: NdiffUpdateEvent) => {
      setNdiffUpdates(prev => [...prev, event]);
      queryClient.invalidateQueries({ queryKey: ['ndiffs'] });
    }
  )

  return (
    <NmapIntegrationUpdatesContext.Provider value={{ scanUpdates, mergeScan, ndiffUpdates, mergeNdiff }}>
      {children}
    </NmapIntegrationUpdatesContext.Provider>
  );
};

export const useNmapIntegrationUpdates = () => {
    const context = useContext(NmapIntegrationUpdatesContext);
    if (!context) throw new Error('useNmapIntegrationUpdates must be used within a NmapIntegrationUpdatesProvider');
    return context;
};
