import React, { useState } from "react";
import {
  Sheet,
  SheetContent,
  SheetHeader,
  SheetFooter,
  SheetTitle,
  SheetDescription,
} from "@/components/ui/sheet";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import type { nmapProfile } from "./types";
import { useUpdateProfile } from "@/components/nmap/profiles/queries";
import { useTranslation } from "react-i18next"

export default function EditProfileSheet({
  open,
  onOpenChange,
  profile,
}: {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  profile: nmapProfile | null;
}) {
  const { t } = useTranslation();
  const [form, setForm] = useState<Partial<nmapProfile>>(profile || {});
  const updateProfileMutation = useUpdateProfile();

  React.useEffect(() => {
    setForm(profile || {});
  }, [profile]);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setForm((prev) => ({ ...prev, [name]: value }));
  };

  const handleTagsChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setForm((prev) => ({ ...prev, tags: e.target.value.split(",").map(t => t.trim()).filter(Boolean) }));
  };

  const handleSave = () => {
    if (!form.id) return;
    updateProfileMutation.mutate(form as nmapProfile, {
      onSuccess: () => onOpenChange(false),
    });
  };

  return (
    <Sheet open={open} onOpenChange={onOpenChange}>
      <SheetContent side="right">
        <SheetHeader>
          <SheetTitle>{t('Edit Profile')}</SheetTitle>
          <SheetDescription>
            {t('Update the fields below and click Save to update the profile')}.
          </SheetDescription>
        </SheetHeader>
        <div className="py-4 flex flex-col gap-4">
          <label className="flex flex-col gap-1">
            {t('Name')}
            <Input name="name" value={form.name || ""} onChange={handleChange} />
          </label>
          <label className="flex flex-col gap-1">
            {t('Parameters')}
            <Textarea className="min-h-64 resize-none" name="parameters" value={form.parameters || ""} onChange={handleChange} rows={3} />
          </label>
          <label className="flex flex-col gap-1">
            {t('Description')}
            <Textarea className="min-h-48 resize-none" name="description" value={form.description || ""} onChange={handleChange} rows={3} />
          </label>
          <label className="flex flex-col gap-1">
            {t('Comma Separated Tags')}
            <Input name="tags" value={Array.isArray(form.tags) ? form.tags.join(", ") : ""} onChange={handleTagsChange} />
          </label>
        </div>
        <SheetFooter>
          <Button variant="outline" onClick={() => onOpenChange(false)} type="button">{t('Cancel')}</Button>
          <Button onClick={handleSave} type="button" disabled={updateProfileMutation.isPending}>
            {updateProfileMutation.isPending ? t('Saving') + '...' : t("Save")}
          </Button>
        </SheetFooter>
      </SheetContent>
    </Sheet>
  );
}