import { Button } from "@/components/ui/button";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import { useState } from "react"
import { columns } from "@/components/nmap/profiles/columns";
import AddProfileSheet from "@/components/nmap/profiles/AddProfileSheet";
import DeleteProfileDialog from "@/components/nmap/profiles/DeleteProfileDialog";
import { nmapProfile } from "@/components/nmap/profiles/types";
import EditProfileSheet from "@/components/nmap/profiles/EditProfileSheet"
import { validateScanHasTarget } from '@/components/nmap/utils';
import { useAddScan } from "@/components/nmap/scans/queries"
import AddEditScheduledScanDialog from "@/components/nmap/scheduledscans/AddEditScheduledScanDialog"
import RunScanDialog from "@/components/nmap/profiles/RunScanDialog"
import { Plus } from "lucide-react";
import { useTranslation } from "react-i18next"
import { useAuth } from "@/contexts/AuthContext";

type ProfilesProps = {
  data: nmapProfile[] | undefined;
  isLoading: boolean;
}

export default function Profiles({ data, isLoading }: ProfilesProps) {
  const { t } = useTranslation();
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [idsToDelete, setIdsToDelete] = useState<number[]>([]);
  const [editSheetOpen, setEditSheetOpen] = useState(false);
  const [profile, setProfile] = useState<nmapProfile | null>(null);
  const [runScanDialogOpen, setRunScanDialogOpen] = useState(false);
  const [addScheduledScanDialogOpen, setAddScheduledScanDialogOpen] = useState(false);
  const addScanMutation = useAddScan();
  const { user } = useAuth();

  const handleRunScan = (nmapProfile: nmapProfile) => {
    setProfile(nmapProfile);
    if (validateScanHasTarget(nmapProfile.parameters)) {
      addScanMutation.mutate({
        title: `${nmapProfile.name}`,
        scan_parameters: nmapProfile.parameters,
      });
    } else {
      setRunScanDialogOpen(true);
    }
  };

  const handleAddScheduledScan = (nmapProfile: nmapProfile) => {
    setProfile(nmapProfile);
    setAddScheduledScanDialogOpen(true);
  }

  const handleEditProfile = (nmapProfile: nmapProfile) => {
    setProfile(nmapProfile);
    setEditSheetOpen(true);
  };

  const handleDelete = (id: number) => {
    setIdsToDelete([id]);
    setDeleteDialogOpen(true);
  }

  const actions = [
    {
      label: "Delete",
      value: "delete",
      onClick: async (selectedRows: any) => {
        setIdsToDelete(selectedRows.map((row: any) => row.id));
        setDeleteDialogOpen(true);
      }
    }
  ]

  return (
    <>
      {!!user?.is_admin &&
        <div className="pb-1 flex">
          <AddProfileSheet>
            <Button>
              <Plus />
              {t('Create Profile')}
            </Button>
          </AddProfileSheet>
        </div>
      }
      {!isLoading && (
        <DataTable 
          columns={columns(
            data || [],
            handleRunScan,
            handleAddScheduledScan,
            handleEditProfile,
            handleDelete,
          )} 
          data={data || []} 
          sorting={[{ id: "Created At", desc: true }]}
          usesRowSelection={true}
          selectedRowActions={actions}
          defaultColumnVisibility={{
            'Times Ran': false,
            'Created At': false,
          }}
        />
      )}
      {isLoading && <DataTableSkeleton />}
      <RunScanDialog
        open={runScanDialogOpen}
        onOpenChange={setRunScanDialogOpen}
        profile={profile as nmapProfile}
      />
      <AddEditScheduledScanDialog
        scanProfile={profile as nmapProfile | undefined}
        open={addScheduledScanDialogOpen}
        onOpenChange={setAddScheduledScanDialogOpen}
        mode="add"
      >
        <span />
      </AddEditScheduledScanDialog>
      <EditProfileSheet
        open={editSheetOpen}
        onOpenChange={setEditSheetOpen}
        profile={profile}
      />
      <DeleteProfileDialog
        ids={idsToDelete}
        open={deleteDialogOpen}
        onOpenChange={(open) => {
          setDeleteDialogOpen(open);
          if (!open) setIdsToDelete([]);
        }}
      >
        <span />
      </DeleteProfileDialog>
    </>
  )
}