import { ColumnDef } from "@tanstack/react-table"
import { Badge } from "@/components/ui/badge"
import { Checkbox } from "@/components/ui/checkbox"
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader"
import { Button } from "@/components/ui/button"
import { 
  MoreHorizontal,
  Trash2,
  Pencil,
  CopyPlus,
  Plus,
  CalendarPlus,
} from "lucide-react"
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"
import { nmapProfile } from "@/components/nmap/profiles/types"
import { useAddProfile } from "@/components/nmap/profiles/queries"
import { useTranslation } from "react-i18next"
import { useAuth } from "@/contexts/AuthContext"
import { userInfo } from "os"

export const columns = (
  data: nmapProfile[],
  handleRunScan: (nmapProfile: nmapProfile) => void,
  handleAddScheduledScan: (nmapProfile: nmapProfile) => void,
  handleEditProfile: (nmapProfile: nmapProfile) => void,
  handleDelete: (id: number) => void,
) : ColumnDef<nmapProfile>[] => [
  {
    id: "select",
    header: ({ table }) => (
      <Checkbox
        checked={
          table.getIsAllPageRowsSelected() ||
          (table.getIsSomePageRowsSelected() && "indeterminate")
        }
        onCheckedChange={(value) => table.toggleAllPageRowsSelected(!!value)}
        aria-label="Select all"
      />
      ),
      cell: ({ row }) => (
        <Checkbox
          checked={row.getIsSelected()}
          onCheckedChange={(value) => row.toggleSelected(!!value)}
          aria-label="Select row"
        />
      ),
      enableSorting: false,
      enableHiding: false,
  },
  {
    accessorKey: "name",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Profile Name" />
    },
    cell: ({ row }) => <span>{row.getValue("name")}</span>,
  },
  {
    id: "parameters",
    accessorKey: "parameters",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Parameters" />
    },
    cell: ({ row }) => (
      <div style={{ overflowWrap: 'break-word', wordBreak: 'normal', whiteSpace: 'pre-line', maxWidth: 400 }}>
        {row.getValue("parameters")}
      </div>
    ),
  },
  {
    id: "description",
    accessorKey: "description",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Description" />
    },
    cell: ({ row }) => (
      <div style={{ overflowWrap: 'break-word', wordBreak: 'normal', whiteSpace: 'pre-line', maxWidth: 700 }}>
        {row.getValue("description") || <span>-</span>}
      </div>
    ),
  },
  {
    id: "tags",
    accessorKey: "tags",
    accessorFn: (row) => Array.isArray(row.tags) ? row.tags.join(" ") : "",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Tags" />
    },
    cell: ({ row }) => {
      const tags = row.original.tags as string[] | undefined;
      if (!tags || tags.length === 0) return <span>-</span>;
      const sortedTags = [...tags].sort((a, b) => a.localeCompare(b));
      return (
        <div style={{ display: "flex", gap: "0.25rem", flexWrap: "wrap" }}>
          {sortedTags.map((tag, index) => (
            <Badge key={index} variant="secondary">{tag}</Badge>
          ))}
        </div>
      );
    },
  },
  {
    id: "Times Ran",
    accessorKey: "times_ran",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Times Ran" />
    },
  },
  {
    id: "Created At",
    accessorKey: "created_at",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Created At" />
    },
    cell: ({ row }) => {
      const { t } = useTranslation();
      const createdAtValue = row.getValue("Created At");

      if (!createdAtValue || (typeof createdAtValue !== "string")) {
        return <div>{t('Unknown')}</div>;
      }

      const createdAt = new Date(createdAtValue);
      return (
        <div>
          {createdAt.toLocaleString()}
        </div>
      );
    },
  },
  {
    id: "actions",
    cell: ({ row }) => {
      const { t } = useTranslation();
      const profile = row.original;
      const addProfileMutation = useAddProfile();
      const { user } = useAuth();

      const handleClone = () => {
        const allNames = data.map((p) => p.name);
        let cloneName = `${profile.name} (Clone)`;

        while (allNames.includes(cloneName)) cloneName += " (Clone)";
        addProfileMutation.mutate({
          name: cloneName,
          parameters: profile.parameters,
          description: profile.description,
          tags: profile.tags || [],
        });
      };

      return (
        <>
          {!!user?.is_admin &&
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="ghost" className="h-8 w-8 p-0">
                  <span className="sr-only">{t('Open menu')}</span>
                  <MoreHorizontal className="h-4 w-4" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="center">
                <DropdownMenuItem onClick={() => handleRunScan(profile)}>
                  <Plus />
                  {t('Run a Scan')}
                </DropdownMenuItem>
                <DropdownMenuItem onClick={() => handleAddScheduledScan(profile)}>
                  <CalendarPlus />
                  {t('Create Scheduled Scan')}
                </DropdownMenuItem>
                <DropdownMenuSeparator />
                <DropdownMenuItem onClick={() => handleEditProfile(profile)}>
                  <Pencil />
                  {t('Edit')}
                </DropdownMenuItem>
                <DropdownMenuItem onClick={handleClone}>
                  <CopyPlus />
                  {t('Clone')}
                </DropdownMenuItem>
                <DropdownMenuItem onClick={() => handleDelete(profile.id)}>
                  <Trash2 />
                  {t('Delete')}
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>
          }
        </>
      );
    },
  },
]